/**
 * Echelonic Planner
 * A JavaScript library for creating interactive floor plans with gateways,
 * geofences, and fingerprinting capabilities.
 *
 * This script contains the core logic for the planner component.
 * All styling has been moved to echelonic.planner.css.
 */
class echelonicPlanner {
    constructor(parentContainer, options = {}) {
        if (!parentContainer) {
            console.error('A parent container element must be provided.');
            return;
        }

        const defaultOptions = {
            map: null,
            mapScale: 0,
            colorScheme: 'blue',
            accentColor: 'deepOrange',
            darkMode: false,
            baseUrl: 'https://api.echelonic.cc',
            authToken: null,
            mqtt: {
                brokerUrl: 'wss://canarsie_locator:LXc7BLWZkqb8V_X@4e716b90940245db882d0093760c1b56.s1.eu.hivemq.cloud:8884/mqtt',
                topic: 'pub',
                tagId: 'F41E57284912'
            },
            endpoints: {
                listGateways: '/gateways',
                createGateway: '/gateways',
                updateGateway: '/gateways/{id}',
                deleteGateway: '/gateways/{id}',
                listGeofences: '/geofences',
                createGeofence: '/geofences',
                updateGeofence: '/geofences/{id}',
                deleteGeofence: '/geofences/{id}',
                getFingerprints: '/fingerprints',
                updateFingerprints: '/fingerprints'
            },
			grid: {
				enabled: true,
				spacingMeters: 1,
				majorEvery: 5,
				lineWidth: 1,
				showLabels: false,
			}
        };

        this.options = { ...defaultOptions, ...options };

        if (!this.options.map || !this.options.mapScale || this.options.mapScale <= 0) {
            console.error('The "map" (URL) and a positive "mapScale" (pixels per meter) must be provided in the options.');
            return;
        }
		this.grid = this.options.grid;

        this.parentContainer = parentContainer;

        // Internal state variables
        this.image = null;
        this.originalImageDimensions = { width: 0, height: 0 };
        this.basePixelsPerMeter = 0;
        this.dragStart = null;
        this.gateways = [];
        this.geofences = [];
        this._apiBuffer = { gateways: null, geofences: null };
        this.selectedGateway = null;
        this.selectedGeofence = null;
        this.selectedNode = null;
        this.selectedPseudoNode = null;
        this.isAddingGateway = false;
        this.isDraggingGateway = false;
        this.isAddingGeofence = false;
        this.isDraggingNode = false;
        this.pendingGatewayPosition = null;
        this.isMeasuring = false;
        this.measureStartPoint = null;
        this.heatmapCanvas = null;
        this.heatmapGradient = null;
        this.heatmapType = 'none';
        this.panX = 0;
        this.panY = 0;
        this.zoomFactor = 1.0;
        this.mousePosition = { x: 0, y: 0 };
        this.gatewayIdCounter = 1;
        this.geofenceIdCounter = 1;
        this.pendingGeofence = null;
        this.dragInitialPosition = null;
        this.lastTouchDistance = null;

        // Fingerprinting State
        this.isFingerprinting = false;
        this.isActivelyRecording = false;
        this.mqttClient = null;
        this.latestRssi = {};
        this.recordingHistory = {};
        this.fingerprints = [];
        this.selectedFingerprintCoords = null;
        this.selectedFingerprintForDeletion = null;
        this.EMA_ALPHA = 0.1;
        this.fingerprintsLoaded = false;

        // Constants
        this.SNAP_DISTANCE = 15;
        this.CLOSE_DISTANCE_THRESHOLD = 20;
        this.HIT_RADIUS_PX = 12;
        this.dragCapture = null;

        this.setupUI();
        this.setupListeners();

        this.loadImageFromUrl(this.options.map);
    }

    // --- Core Methods & Helpers ---
    _hitNodeAt(geofence, mouseX, mouseY, radius = 10) {
        return geofence.nodes.find(node => {
            const x = node.x * this.originalImageDimensions.width * this.zoomFactor + this.panX;
            const y = node.y * this.originalImageDimensions.height * this.zoomFactor + this.panY;
            return Math.hypot(mouseX - x, mouseY - y) <= radius;
        }) || null;
    }

    _hitPseudoAt(geofence, mouseX, mouseY, radius = 10) {
        const pseudo = this.updatePseudoNodes(geofence.nodes);
        return pseudo.find(p => {
            const x = p.x * this.originalImageDimensions.width * this.zoomFactor + this.panX;
            const y = p.y * this.originalImageDimensions.height * this.zoomFactor + this.panY;
            return Math.hypot(mouseX - x, mouseY - y) <= radius;
        }) || null;
    }

    createElement(tag, classes = [], content = '') {
        const el = document.createElement(tag);
        if (classes.length > 0) el.className = classes.join(' ');
        if (content) el.innerHTML = content;
        return el;
    }

    getPanX() { return this.panX ?? this.offsetX ?? this.viewOffsetX ?? 0; }
    getPanY() { return this.panY ?? this.offsetY ?? this.viewOffsetY ?? 0; }

    screenToNorm(sx, sy) {
        return {
            x: (sx - this.panX) / (this.originalImageDimensions.width * this.zoomFactor),
            y: (sy - this.panY) / (this.originalImageDimensions.height * this.zoomFactor),
        };
    }

    screenFromNorm(pt) {
        return {
            x: pt.x * this.originalImageDimensions.width * this.zoomFactor + this.panX,
            y: pt.y * this.originalImageDimensions.height * this.zoomFactor + this.panY
        };
    }

    _calculateMedian(arr = []) {
        if (arr.length === 0) return NaN;
        const sorted = [...arr].sort((a, b) => a - b);
        const mid = Math.floor(sorted.length / 2);
        return sorted.length % 2 !== 0 ? sorted[mid] : (sorted[mid - 1] + sorted[mid]) / 2;
    }

    _calculateEMA(arr = [], alpha = this.EMA_ALPHA) {
        if (arr.length === 0) return NaN;
        let ema = arr[0];
        for (let i = 1; i < arr.length; i++) {
            ema = alpha * arr[i] + (1 - alpha) * ema;
        }
        return ema;
    }

    // --- UI Setup ---
    handleWheel(e) {
        e.preventDefault();
        const zoomSpeed = 1.1;
        const zoomFactor = e.deltaY < 0 ? zoomSpeed : 1 / zoomSpeed;
        const rect = this.canvas.getBoundingClientRect();
        const anchorX = e.clientX - rect.left;
        const anchorY = e.clientY - rect.top;
        this.zoomBy(zoomFactor, anchorX, anchorY);
    }

    handleTouchStart(e) {
        if (e.touches.length === 2) {
            e.preventDefault();
            const touch1 = e.touches[0];
            const touch2 = e.touches[1];
            this.lastTouchDistance = Math.hypot(touch1.clientX - touch2.clientX, touch1.clientY - touch2.clientY);
        }
    }

    handleTouchMove(e) {
        if (e.touches.length === 2 && this.lastTouchDistance) {
            e.preventDefault();
            const touch1 = e.touches[0];
            const touch2 = e.touches[1];
            const newDist = Math.hypot(touch1.clientX - touch2.clientX, touch1.clientY - touch2.clientY);
            const zoomFactor = newDist / this.lastTouchDistance;
            const rect = this.canvas.getBoundingClientRect();
            const anchorX = ((touch1.clientX + touch2.clientX) / 2) - rect.left;
            const anchorY = ((touch1.clientY + touch2.clientY) / 2) - rect.top;
            this.zoomBy(zoomFactor, anchorX, anchorY);
            this.lastTouchDistance = newDist;
        }
    }

    handleTouchEnd(e) {
        if (e.touches.length < 2) {
            this.lastTouchDistance = null;
        }
    }

    setupUI() {
        // Apply theme classes and data attributes for styling via CSS
        this.parentContainer.className = 'e-planner-container';
        this.parentContainer.classList.toggle('e-planner-dark-mode', this.options.darkMode);
        this.parentContainer.dataset.colorScheme = this.options.colorScheme;
        this.parentContainer.dataset.accentColor = this.options.accentColor;
        this.parentContainer.innerHTML = '';

        const canvasContainer = this.createElement('div', ['e-planner-canvas-container']);
        this.canvas = this.createElement('canvas');
        this.ctx = this.canvas.getContext('2d');
        canvasContainer.appendChild(this.canvas);
        this.canvasContainer = canvasContainer;
        this.measureTooltip = this.createElement('div', ['e-measure-tooltip', 'hidden'], '0.00 m');
        this.canvasContainer.appendChild(this.measureTooltip);

        const topOverlay = this.createElement('div', ['e-overlay', 'e-overlay-top']);
        const topLeftGroup = this.createElement('div', ['e-panel-group']);
        const topRightGroup = this.createElement('div', ['e-panel-group']);
        topOverlay.appendChild(topLeftGroup);
        topOverlay.appendChild(topRightGroup);

        this.gatewayControls = this.createElement('div', ['e-planner-panel', 'wrap', 'hidden']);
        this.addGatewayBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-inverted'], '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="inherit"><path d="M198-278q-57-57-87.5-129.5T80-560q0-80 30.5-152.5T198-842l48 48q-47 47-72.5 107.5T148-560q0 66 25.5 126.5T246-326l-48 48Zm92-92q-38-38-58-87t-20-103q0-54 20-103t58-87l48 48q-29 29-43.5 65.5T280-560q0 40 14.5 76.5T338-418l-48 48Zm150 250v-348q-27-12-43.5-37T380-560q0-42 29-71t71-29q42 0 71 29t29 71q0 30-16.5 55T520-468v348h-80Zm230-250-48-48q29-29 43.5-65.5T680-560q0-40-14.5-76.5T622-702l48-48q38 38 58 87t20 103q0 54-20 103t-58 87Zm92 92-48-48q47-47 72.5-107.5T812-560q0-66-25.5-126.5T714-794l48-48q57 57 87.5 129.5T880-560q0 80-30.5 152.5T762-278Z"/></svg>');
        this.addGeofenceBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-primary'], '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="inherit"><path d="M160-160q-33 0-56.5-23.5T80-240v-120h80v120h120v80H160Zm192-160 40-204-72 28v136h-80v-188l158-68q35-15 51.5-19.5T480-640q21 0 39 11t29 29l40 64 8 12q4 6 9 11-49 32-81 82.5T483-320H352Zm188-340q-33 0-56.5-23.5T460-740q0-33 23.5-56.5T540-820q33 0 56.5 23.5T620-740q0 33-23.5 56.5T540-660ZM80-760v-120q0-33 23.5-56.5T160-960h120v80H160v120H80Zm720 0v-120H680v-80h120q33 0 56.5 23.5T880-880v120h-80ZM760-80q-83 0-141.5-58.5T560-280q0-83 58.5-141.5T760-480q83 0 141.5 58.5T960-280q0 83-58.5 141.5T760-80Zm-20-160h40v-160h-40v160Zm20 80q8 0 14-6t6-14q0-8-6-14t-14-6q-8 0-14 6t-6 14q0 8 6 14t14 6Z"/></svg>');
        this.measureBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral'], '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="inherit"><path d="M200-160v-340q0-142 99-241t241-99q142 0 241 99t99 241q0 142-99 241t-241 99H200Zm80-80h260q108 0 184-76t76-184q0-108-76-184t-184-76q-108 0-184 76t-76 184v260Zm260-120q58 0 99-41t41-99q0-58-41-99t-99-41q-58 0-99 41t-41 99q0 58 41 99t99 41Zm0-80q-25 0-42.5-17.5T480-500q0-25 17.5-42.5T540-560q25 0 42.5 17.5T600-500q0 25-17.5 42.5T540-440ZM80-160v-200h80v200H80Zm460-340Z"/></svg>');
        this.environmentSelect = this.createElement('select', ['e-planner-select']);
        this.environmentSelect.innerHTML = `<option value="none">Heatmap: None</option><option value="office">Office</option><option value="residential">Residential</option><option value="industrial">Industrial</option><option value="outdoors">Outdoors</option>`;
        this.fingerprintBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-success'], '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="inherit"><path d="M481-781q106 0 200 45.5T838-604q7 9 4.5 16t-8.5 12q-6 5-14 4.5t-14-8.5q-55-78-141.5-119.5T481-741q-97 0-182 41.5T158-580q-6 9-14 10t-14-4q-7-5-8.5-12.5T126-602q62-85 155.5-132T481-781Zm0 94q135 0 232 90t97 223q0 50-35.5 83.5T688-257q-51 0-87.5-33.5T564-374q0-33-24.5-55.5T481-452q-34 0-58.5 22.5T398-374q0 97 57.5 162T604-121q9 3 12 10t1 15q-2 7-8 12t-15 3q-104-26-170-103.5T358-374q0-50 36-84t87-34q51 0 87 34t36 84q0 33 25 55.5t59 22.5q34 0 58-22.5t24-55.5q0-116-85-195t-203-79q-118 0-203 79t-85 194q0 24 4.5 60t21.5 84q3 9-.5 16T208-205q-8 3-15.5-.5T182-217q-15-39-21.5-77.5T154-374q0-133 96.5-223T481-687Zm0-192q64 0 125 15.5T724-819q9 5 10.5 12t-1.5 14q-3 7-10 11t-17-1q-53-27-109.5-41.5T481-839q-58 0-114 13.5T260-783q-8 5-16 2.5T232-791q-4-8-2-14.5t10-11.5q56-30 117-46t124-16Zm0 289q93 0 160 62.5T708-374q0 9-5.5 14.5T688-354q-8 0-14-5.5t-6-14.5q0-75-55.5-125.5T481-550q-76 0-130.5 50.5T296-374q0 81 28 137.5T406-123q6 6 6 14t-6 14q-6 6-14 6t-14-6q-59-62-90.5-126.5T256-374q0-91 66-153.5T481-590Zm-1 196q9 0 14.5 6t5.5 14q0 75 54 123t126 48q6 0 17-1t23-3q9-2 15.5 2.5T744-191q2 8-3 14t-13 8q-18 5-31.5 5.5t-16.5.5q-89 0-154.5-60T460-374q0-8 5.5-14t14.5-6Z"/></svg>');

        if (typeof mqtt === 'undefined' || mqtt === null) {
            console.warn('MQTT.js library not found. Hiding fingerprinting functionality.');
            this.fingerprintBtn.classList.add('hidden');
        }

        this.gatewayControls.appendChild(this.addGatewayBtn);
        this.gatewayControls.appendChild(this.addGeofenceBtn);
        this.gatewayControls.appendChild(this.fingerprintBtn);
        this.gatewayControls.appendChild(this.measureBtn);
        this.gatewayControls.appendChild(this.environmentSelect);
        topLeftGroup.appendChild(this.gatewayControls);

        this.selectionControls = this.createElement('div', ['e-planner-panel', 'wrap', 'hidden', 'gw-controls']);
        this.editGatewayBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral', 'hidden'], 'Edit Gateway');
        this.removeGatewayBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-danger', 'hidden'], 'Remove Gateway');
        this.editGeofenceBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral', 'hidden'], 'Edit Geofence');
        this.removeGeofenceBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-danger', 'hidden'], 'Remove Geofence');
        this.removeNodeBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-danger', 'hidden'], 'Remove Node');
        this.finishGeofenceBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-success', 'hidden'], 'Confirm');
        this.cancelGeofenceBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral', 'hidden'], 'Cancel');
        this.selectionControls.appendChild(this.editGatewayBtn); this.selectionControls.appendChild(this.removeGatewayBtn);
        this.selectionControls.appendChild(this.editGeofenceBtn); this.selectionControls.appendChild(this.removeGeofenceBtn);
        this.selectionControls.appendChild(this.removeNodeBtn);
        this.selectionControls.appendChild(this.finishGeofenceBtn);
        this.selectionControls.appendChild(this.cancelGeofenceBtn);
        topRightGroup.appendChild(this.selectionControls);

        // --- Fingerprint UI ---
        this.fingerprintPanel = this.createElement('div', ['e-planner-panel', 'e-fingerprint-panel', 'hidden']);
        this.fingerprintRefTag = this.createElement('input', ['e-planner-input', 'e-planner-ref-tag']);
        this.fingerprintRefTag.placeholder = "Enter reference tag MAC";
        this.fingerprintRefTag.value = this.options.mqtt.tagId;
        this.fingerprintStatusEl = this.createElement('p', ['e-fingerprint-status']);
        this.rssiReadingsEl = this.createElement('div', ['e-rssi-readings']);
        const fingerprintInfo = this.createElement('div', ['e-fingerprint-info']);
        this.selectedPointEl = this.createElement('span', [], 'Selected: None');
        this.fingerprintCountEl = this.createElement('span', [], '0 recorded');
        fingerprintInfo.appendChild(this.selectedPointEl); fingerprintInfo.appendChild(this.fingerprintCountEl);
        const fpButtons = this.createElement('div', ['e-planner-button-group']);
        this.recordFingerprintBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-success'], 'Record'); this.recordFingerprintBtn.disabled = true;
        this.deleteFingerprintBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-danger'], 'Delete'); this.deleteFingerprintBtn.disabled = true;
        this.saveFingerprintsBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-primary'], 'Update'); this.saveFingerprintsBtn.disabled = true;
        this.exportFingerprintsBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-primary'], 'Download'); this.exportFingerprintsBtn.disabled = true;
        this.cancelFingerprintBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral'], 'Close');
        fpButtons.appendChild(this.recordFingerprintBtn);
		this.recordFingerprintBtn.style.minWidth = "81px";
        fpButtons.appendChild(this.deleteFingerprintBtn);
        fpButtons.appendChild(this.saveFingerprintsBtn);
        fpButtons.appendChild(this.exportFingerprintsBtn);
        fpButtons.appendChild(this.cancelFingerprintBtn);
        this.fingerprintPanel.appendChild(this.fingerprintRefTag);
        this.fingerprintPanel.appendChild(this.fingerprintStatusEl);
        this.fingerprintPanel.appendChild(this.rssiReadingsEl);
        this.fingerprintPanel.appendChild(fingerprintInfo);
        this.fingerprintPanel.appendChild(fpButtons);
        topRightGroup.appendChild(this.fingerprintPanel);

        const bottomCenterOverlay = this.createElement('div', ['e-overlay', 'e-overlay-bottom-center']);
        const zoomControlsDiv = this.createElement('div', ['e-planner-panel']);
        this.zoomInBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral'], '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="inherit"><path d="M440-440H200v-80h240v-240h80v240h240v80H520v240h-80v-240Z"/></svg>');
        this.zoomOutBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral'], '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="inherit"><path d="M200-440v-80h560v80H200Z"/></svg>');
        this.zoomToFitBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral'], '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="inherit"><path d="M800-600v-120H680v-80h120q33 0 56.5 23.5T880-720v120h-80Zm-720 0v-120q0-33 23.5-56.5T160-800h120v80H160v120H80Zm600 440v-80h120v-120h80v120q0 33-23.5 56.5T800-160H680Zm-520 0q-33 0-56.5-23.5T80-240v-120h80v120h120v80H160Zm80-160v-320h480v320H240Zm80-80h320v-160H320v160Zm0 0v-160 160Z"/></svg>');
        zoomControlsDiv.appendChild(this.zoomInBtn); zoomControlsDiv.appendChild(this.zoomOutBtn); zoomControlsDiv.appendChild(this.zoomToFitBtn);
        bottomCenterOverlay.appendChild(zoomControlsDiv);

        canvasContainer.appendChild(topOverlay); canvasContainer.appendChild(bottomCenterOverlay);
        this.parentContainer.appendChild(canvasContainer);

        // Modals
        this.gatewayModal = this.createElement('div', ['e-planner-modal', 'hidden']);
        const gatewayModalContent = this.createElement('div', ['e-planner-modal-content']);
        this.gatewayModalTitle = this.createElement('h2', ['e-planner-modal-title'], 'Add Gateway');
        this.macAddressInput = this.createElement('input', ['e-planner-input']); this.macAddressInput.placeholder = "MAC Address (e.g., AABBCCDD1122)";
        this.gatewayNameInput = this.createElement('input', ['e-planner-input']); this.gatewayNameInput.placeholder = "Name (optional)";
        const gatewayModalButtons = this.createElement('div', ['e-planner-modal-buttons']);
        this.cancelModalBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral'], 'Cancel');
        this.saveGatewayBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-primary'], 'Save Gateway');
        gatewayModalButtons.appendChild(this.cancelModalBtn); gatewayModalButtons.appendChild(this.saveGatewayBtn);
        gatewayModalContent.appendChild(this.gatewayModalTitle); gatewayModalContent.appendChild(this.macAddressInput);
        gatewayModalContent.appendChild(this.gatewayNameInput);
        gatewayModalContent.appendChild(gatewayModalButtons);
        this.gatewayModal.appendChild(gatewayModalContent);

        this.geofenceModal = this.createElement('div', ['e-planner-modal', 'hidden']);
        const geofenceModalContent = this.createElement('div', ['e-planner-modal-content']);
        this.geofenceModalTitle = this.createElement('h2', ['e-planner-modal-title'], 'Add Geofence');
        this.geofenceNameInput = this.createElement('input', ['e-planner-input']); this.geofenceNameInput.placeholder = "Geofence Name";
        const geofenceStatusDiv = this.createElement('div', ['e-planner-radio-group']);
        geofenceStatusDiv.innerHTML = `<span>Status:</span><label><input type="radio" name="geofenceStatus" value="active" checked> Active</label><label><input type="radio" name="geofenceStatus" value="inactive"> Inactive</label>`;
        const geofenceModalButtons = this.createElement('div', ['e-planner-modal-buttons']);
        this.cancelGeofenceModalBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral'], 'Cancel');
        this.saveGeofenceBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-primary'], 'Save Geofence');
        geofenceModalButtons.appendChild(this.cancelGeofenceModalBtn); geofenceModalButtons.appendChild(this.saveGeofenceBtn);
        geofenceModalContent.appendChild(this.geofenceModalTitle); geofenceModalContent.appendChild(this.geofenceNameInput);
        geofenceModalContent.appendChild(geofenceStatusDiv); geofenceModalContent.appendChild(geofenceModalButtons);
        this.geofenceModal.appendChild(geofenceModalContent);

        this.parentContainer.appendChild(this.gatewayModal); this.parentContainer.appendChild(this.geofenceModal);

        this.fingerprintWarningModal = this.createElement('div', ['e-planner-modal', 'hidden']);
        const warningModalContent = this.createElement('div', ['e-planner-modal-content']);
        warningModalContent.innerHTML = `
            <h2 class="e-planner-modal-title e-planner-modal-title--danger">Update Gateway Position?</h2>
            <p class="e-planner-modal-message">
                Moving a gateway will invalidate all existing fingerprints. You will need to perform fingerprint collection again.
            </p>
        `;
        const warningModalButtons = this.createElement('div', ['e-planner-modal-buttons']);
        this.cancelMoveBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-neutral'], 'Cancel');
        this.confirmMoveBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-danger'], 'Confirm & Move');
        warningModalButtons.appendChild(this.cancelMoveBtn);
        warningModalButtons.appendChild(this.confirmMoveBtn);
        warningModalContent.appendChild(warningModalButtons);
        this.fingerprintWarningModal.appendChild(warningModalContent);
        this.parentContainer.appendChild(this.fingerprintWarningModal);

        this.alertModal = this.createElement('div', ['e-planner-modal', 'hidden']);
        const alertModalContent = this.createElement('div', ['e-planner-modal-content']);
        this.alertModalTitle = this.createElement('h2', ['e-planner-modal-title'], 'Notification');
        this.alertModalMessage = this.createElement('p', ['e-planner-modal-message']);
        const alertModalButtons = this.createElement('div', ['e-planner-modal-buttons']);
        this.alertModalCloseBtn = this.createElement('button', ['e-planner-button', 'e-planner-button-primary'], 'OK');
        alertModalButtons.appendChild(this.alertModalCloseBtn);
        alertModalContent.appendChild(this.alertModalTitle);
        alertModalContent.appendChild(this.alertModalMessage);
        alertModalContent.appendChild(alertModalButtons);

        this.alertModal.appendChild(alertModalContent);
        this.parentContainer.appendChild(this.alertModal);

        this.alertModalCloseBtn.onclick = () => this.alertModal.classList.add("hidden");
    }

    // ... (The rest of the class methods from the original script are unchanged)
	// Make sure to copy all methods from setupListeners() to the end of the class
	// from the original script provided, as they handle the core logic.
	// For brevity, they are omitted here but are required for functionality.
	
    setupListeners() {
    // Main Controls
    this.addGatewayBtn.addEventListener('click', this.addGateway.bind(this));
    this.addGeofenceBtn.addEventListener('click', this.addGeofence.bind(this));
    this.finishGeofenceBtn.addEventListener('click', this.finishGeofence.bind(this));
    this.cancelGeofenceBtn.addEventListener('click', this.cancelGeofenceModal.bind(this));
    this.measureBtn.addEventListener('click', this.toggleMeasure.bind(this));
    this.environmentSelect.addEventListener('change', (e) => { this.heatmapType = e.target.value; this.redrawCanvas(); });

    // Fingerprinting Controls
    this.fingerprintBtn.addEventListener('click', this.toggleFingerprinting.bind(this));
    this.deleteFingerprintBtn.addEventListener('click', this.deleteSelectedFingerprint.bind(this));
    this.recordFingerprintBtn.addEventListener('click', this.recordFingerprint.bind(this));
    this.saveFingerprintsBtn.addEventListener('click', this.saveFingerprintsToServer.bind(this));
    this.exportFingerprintsBtn.addEventListener('click', this.exportFingerprints.bind(this));
    this.cancelFingerprintBtn.addEventListener('click', this.toggleFingerprinting.bind(this));

    // Selection Controls
    this.editGatewayBtn.addEventListener('click', this.editGateway.bind(this));
    this.removeGatewayBtn.addEventListener('click', this.removeGateway.bind(this));
    this.editGeofenceBtn.addEventListener('click', this.editGeofence.bind(this));
    this.removeGeofenceBtn.addEventListener('click', this.removeGeofence.bind(this));
    this.removeNodeBtn.addEventListener('click', this.removeNode.bind(this));

    // Canvas Interactions (Corrected Block)
    this.canvas.addEventListener('mousedown', this.handleMouseDown.bind(this));
    this.canvas.addEventListener('mouseup', this.handleMouseUp.bind(this));
    this.canvas.addEventListener('mousemove', this.handleMouseMove.bind(this));
    this.canvas.addEventListener('click', this.handleMouseClick.bind(this));
    this.canvas.addEventListener('wheel', this.handleWheel.bind(this));
    this.canvas.addEventListener('touchstart', this.handleTouchStart.bind(this));
    this.canvas.addEventListener('touchmove', this.handleTouchMove.bind(this));
    this.canvas.addEventListener('touchend', this.handleTouchEnd.bind(this));

    // Modal Buttons
    this.saveGatewayBtn.addEventListener('click', this.saveGateway.bind(this));
    this.cancelModalBtn.addEventListener('click', this.cancelModal.bind(this));
    this.saveGeofenceBtn.addEventListener('click', this.saveGeofence.bind(this));
    this.cancelGeofenceModalBtn.addEventListener('click', this.cancelGeofenceModal.bind(this));
    this.confirmMoveBtn.addEventListener('click', this.handleConfirmGatewayMove.bind(this));
    this.cancelMoveBtn.addEventListener('click', this.handleCancelGatewayMove.bind(this));

    // Zoom Controls
    this.zoomInBtn.addEventListener('click', this.zoomIn.bind(this));
    this.zoomOutBtn.addEventListener('click', this.zoomOut.bind(this));
    this.zoomToFitBtn.addEventListener('click', this.zoomToFit.bind(this));

    // Global Listener
    window.addEventListener('resize', this.redrawCanvas.bind(this));
}
    
    // --- Initial Load ---
    loadImageFromUrl(url) {
        this.image = new Image();
        this.image.crossOrigin = 'Anonymous'; 
        
        this.image.onload = () => {
            this.originalImageDimensions.width = this.image.width;
            this.originalImageDimensions.height = this.image.height;
            this.basePixelsPerMeter = this.options.mapScale;
            if (this._reifyBufferedAPIData) this._reifyBufferedAPIData();
            console.log(`Image loaded: ${this.originalImageDimensions.width}x${this.originalImageDimensions.height}px`);
            console.log(`Scale set to: ${this.basePixelsPerMeter} pixels per meter.`);
            this.gatewayControls.classList.remove('hidden');
            this.zoomToFit();
            if (this.baseUrl) {
                this.loadGatewaysFromAPI();
                this.loadGeofencesFromAPI();
				// --- Load external fingerprints if they haven't been loaded yet ---
				if (!this.fingerprintsLoaded) {
					this.loadExternalFingerprints().then(() => {
						this.fingerprintsLoaded = true; // Mark as loaded to prevent re-fetching
					});
				}
            }
        };

        this.image.onerror = () => {
            console.error(`Failed to load image from URL: ${url}`);
            this.parentContainer.innerHTML = `<div class="e-planner-error">Error: Could not load the map image.</div>`;
        };
        this.image.src = url;
    }
    
    // --- Mode Handlers (Gateway, Geofence, Measure) ---
    _enterMode(mode) {
		// Clear all temporary and selection states
        this.isActivelyRecording = false; 
        this.selectedGateway = null; 
		this.selectedGeofence = null;
        this.selectedNode = null; 
		this.selectedPseudoNode = null;
        this.selectedFingerprintCoords = null;
        this.pendingGeofence = null; // --- Ensure this is cleared ---
		
        this.isAddingGateway = (mode === 'gateway');
        this.isAddingGeofence = (mode === 'geofence');
        this.isMeasuring = (mode === 'measure');
        this.isFingerprinting = (mode === 'fingerprint');

        
        
        if (this.isFingerprinting) {
            this.connectMqtt();
        } else {
            this.disconnectMqtt();
        }

        this.canvas.classList.toggle('crosshair', this.isAddingGateway || this.isAddingGeofence || this.isMeasuring || this.isFingerprinting);
        this.updateButtonVisibility();
        this.redrawCanvas();
    }

    addGateway() { this._enterMode('gateway'); }
    editGateway() { if (this.selectedGateway) this.showGatewayModal(this.selectedGateway); }
    async removeGateway() {
        if (!this.selectedGateway) return;
        const gw = this.selectedGateway;
        try {
            if (gw.id) await this.deleteGatewayAPI(gw);
            this.gateways = this.gateways.filter(g => g !== gw);
            console.log('Gateway Removed:', gw);
        } catch (e) {
            console.error("Failed to delete gateway:", e);
            this.showAlert("Failed to delete gateway: " + e.message);
        } finally {
            this.selectedGateway = null;
            this.redrawCanvas();
            this.updateButtonVisibility();
        }
    }
	
	async handleConfirmGatewayMove() {
        if (!this.selectedGateway) return;
        try {
            await this.updateGatewayAPI(this.selectedGateway);
            console.log('Gateway position updated.');
        } catch (err) {
            console.error('Failed to update gateway position:', err);
            this.showAlert('Failed to save new position: ' + err.message);
            // If the API fails, revert the move just like a cancel.
            this.handleCancelGatewayMove();
            return;
        } finally {
            this.fingerprintWarningModal.classList.add('hidden');
            this.dragStart = null;
            this.isDraggingGateway = false;
            this.canvas.classList.remove('grabbing');
            this.dragInitialPosition = null;
            this.selectedGateway = null; // Deselect after move
            this.updateButtonVisibility();
            this.redrawCanvas();
        }
    }

    handleCancelGatewayMove() {
        if (this.selectedGateway && this.dragInitialPosition) {
            // Snap the gateway back to its original position
            this.selectedGateway.x = this.dragInitialPosition.x;
            this.selectedGateway.y = this.dragInitialPosition.y;
        }
        this.fingerprintWarningModal.classList.add('hidden');
        this.dragStart = null;
        this.isDraggingGateway = false;
        this.canvas.classList.remove('grabbing');
        this.dragInitialPosition = null;
        this.selectedGateway = null; // Deselect after cancelling
        this.updateButtonVisibility();
        this.redrawCanvas();
    }
    
    addGeofence() {
        this._enterMode('geofence');
        // Create a temporary object instead of modifying the main array
        this.pendingGeofence = { nodes: [] }; 
        console.log("Started new geofence in pending state");
        this.updateButtonVisibility();
    }
    finishGeofence() {
        if (this.isAddingGeofence && this.pendingGeofence) {
            if (this.pendingGeofence.nodes.length >= 3) {
                // We keep the pending geofence object and show the modal.
                // _enterMode(null) will be called by the modal's save/cancel actions.
                this.showGeofenceModal();
            } else {
                this.showAlert("A geofence must have at least 3 nodes.");
            }
        }
        this.updateButtonVisibility();
    }
    editGeofence() { if(this.selectedGeofence) this.showGeofenceModal(this.selectedGeofence); }
    async removeGeofence() {
        if (!this.selectedGeofence) return;
        const f = this.selectedGeofence;
        try {
            if (f.id) await this.deleteGeofenceAPI(f);
            this.geofences = this.geofences.filter(p => p !== f);
            this.selectedGeofence = null; this.selectedNode = null; this.selectedPseudoNode = null;
            this.redrawCanvas();
            console.log('Geofence Removed:', f);
        } catch (e) {
            console.error('Failed to delete geofence:', e);
            this.showAlert('Failed to delete geofence: ' + e.message);
        } finally {
            this.updateButtonVisibility();
        }
    }
    
    async removeNode() {
      if (!this.selectedGeofence || !this.selectedNode) return;
      const f = this.selectedGeofence, node = this.selectedNode, prevNodes = f.nodes.slice();
      f.nodes = f.nodes.filter(n => n !== node);
      this.selectedNode = null;
      this.removeNodeBtn.classList.add('hidden');
      this.redrawCanvas();
      if (f.nodes.length < 3) {
        this.showAlert('A geofence must have at least 3 nodes. Delete the geofence instead.');
        f.nodes = prevNodes; this.redrawCanvas(); this.updateButtonVisibility(); return;
      }
      try {
        if (f.id) await this.updateGeofenceAPI(f);
        console.log('Node Removed');
      } catch (e) {
        console.error('Failed to update geofence after node removal:', e);
        this.showAlert('Failed to update geofence: ' + e.message);
        f.nodes = prevNodes; this.redrawCanvas();
      } finally {
        this.updateButtonVisibility();
      }
    }

    toggleMeasure() {
        if(this.isMeasuring) {
            this._enterMode(null);
            if (this.measureTooltip) this.measureTooltip.classList.add('hidden');
            console.log("Tape measure deactivated.");
        } else {
            this._enterMode('measure');
            console.log("Tape measure activated.");
        }
    }
    
    // --- Fingerprinting Methods ---
    
    // --- MODIFICATION: Method functionality updated ---
    toggleFingerprinting() {
        if (this.isFingerprinting) {
            this._enterMode(null);
            this.fingerprintBtn.innerHTML = '<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="inherit"><path d="M481-781q106 0 200 45.5T838-604q7 9 4.5 16t-8.5 12q-6 5-14 4.5t-14-8.5q-55-78-141.5-119.5T481-741q-97 0-182 41.5T158-580q-6 9-14 10t-14-4q-7-5-8.5-12.5T126-602q62-85 155.5-132T481-781Zm0 94q135 0 232 90t97 223q0 50-35.5 83.5T688-257q-51 0-87.5-33.5T564-374q0-33-24.5-55.5T481-452q-34 0-58.5 22.5T398-374q0 97 57.5 162T604-121q9 3 12 10t1 15q-2 7-8 12t-15 3q-104-26-170-103.5T358-374q0-50 36-84t87-34q51 0 87 34t36 84q0 33 25 55.5t59 22.5q34 0 58-22.5t24-55.5q0-116-85-195t-203-79q-118 0-203 79t-85 194q0 24 4.5 60t21.5 84q3 9-.5 16T208-205q-8 3-15.5-.5T182-217q-15-39-21.5-77.5T154-374q0-133 96.5-223T481-687Zm0-192q64 0 125 15.5T724-819q9 5 10.5 12t-1.5 14q-3 7-10 11t-17-1q-53-27-109.5-41.5T481-839q-58 0-114 13.5T260-783q-8 5-16 2.5T232-791q-4-8-2-14.5t10-11.5q56-30 117-46t124-16Zm0 289q93 0 160 62.5T708-374q0 9-5.5 14.5T688-354q-8 0-14-5.5t-6-14.5q0-75-55.5-125.5T481-550q-76 0-130.5 50.5T296-374q0 81 28 137.5T406-123q6 6 6 14t-6 14q-6 6-14 6t-14-6q-59-62-90.5-126.5T256-374q0-91 66-153.5T481-590Zm-1 196q9 0 14.5 6t5.5 14q0 75 54 123t126 48q6 0 17-1t23-3q9-2 15.5 2.5T744-191q2 8-3 14t-13 8q-18 5-31.5 5.5t-16.5.5q-89 0-154.5-60T460-374q0-8 5.5-14t14.5-6Z"/></svg>';
            this.fingerprintBtn.classList.remove('e-planner-button-danger');
        } else {
            this._enterMode('fingerprint');
            this.fingerprintBtn.textContent = 'Stop Collecting';
            this.fingerprintBtn.classList.add('e-planner-button-danger');
        }
    }

	async loadExternalFingerprints() {
	  try {
		const raw = await this.apiFetch(this.endpoints.getFingerprints, { method: 'GET' });
		const fileContent = raw;
            let data;
            try {
                data = JSON.parse(fileContent);
            } catch (e) {
                console.error("Could not parse fingerprint.js as JSON. Please ensure it is a valid JSON file.", e);
                return;
            }

            const externalFingerprintsInMeters = data.fingerprints;

            if (!Array.isArray(externalFingerprintsInMeters)) {
                console.warn("Data loaded from fingerprint.js does not contain a 'fingerprints' array.");
                return;
            }

            // console.log(`Loaded ${externalFingerprintsInMeters.length} fingerprints from ${url}.`);

            const newFingerprints = externalFingerprintsInMeters.map(fp => {
                //Validate and convert from meters to normalized coordinates
                if (fp.location && typeof fp.location.x === 'number' && typeof fp.location.y === 'number') {
                    return {
                        location: this.metersToNorm(fp.location),
                        readings: fp.readings || {}
                    };
                }
                return null;
            }).filter(fp => fp !== null);

            //Combine with existing fingerprints, preventing duplicates
            const existingLocations = new Set(this.fingerprints.map(fp => `${fp.location.x.toFixed(6)},${fp.location.y.toFixed(6)}`));
            const uniqueNewFingerprints = newFingerprints.filter(fp => {
                const key = `${fp.location.x.toFixed(6)},${fp.location.y.toFixed(6)}`;
                if (existingLocations.has(key)) {
                    return false; // It's a duplicate, filter it out
                }
                existingLocations.add(key);
                return true;
            });

            this.fingerprints.push(...uniqueNewFingerprints);
            console.log(`Added ${uniqueNewFingerprints.length} unique new fingerprints. Total is now ${this.fingerprints.length}.`);

            //Update UI elements
			this.fingerprintCountEl.textContent = `${this.fingerprints.length} recorded`;
			this.exportFingerprintsBtn.disabled = this.fingerprints.length === 0;
			this.saveFingerprintsBtn.disabled = this.fingerprints.length === 0;
			this.redrawCanvas();
	  } catch (err) { console.error('Failed to load fingerprints:', err); }
	}
    
    connectMqtt() {
        const { brokerUrl, topic } = this.options.mqtt;
        // Get the tagId directly from the input field's current value.
        const tagId = this.fingerprintRefTag.value;

        if (!brokerUrl || !topic || !tagId) {
            this.showAlert('MQTT connection details (brokerUrl, topic) and a reference tag ID must be provided.');
            this.toggleFingerprinting();
            return;
        }

        this.fingerprintStatusEl.textContent = `Connecting...`;
        
        try {
            const url = new URL(brokerUrl);
            const mqttOptions = { clientId: 'echelonic_planner_' + Math.random().toString(16).substr(2, 8), username: url.username, password: url.password, reconnectPeriod: 5000, };
            this.mqttClient = mqtt.connect(brokerUrl, mqttOptions);
        } catch(e) {
            console.error("Invalid MQTT Broker URL", e); this.fingerprintStatusEl.textContent = `Error: Invalid Broker URL.`; return;
        }

        this.mqttClient.on('connect', () => {
            this.fingerprintStatusEl.textContent = `Subscribing...`;
            this.mqttClient.subscribe(topic, (err) => {
                if (err) { this.fingerprintStatusEl.textContent = `Subscription error: ${err.message}`; this.mqttClient.end(); } 
                else { this.fingerprintStatusEl.textContent = `Waiting for readings...`; }
            });
        });

        this.mqttClient.on('message', (receivedTopic, message) => {
            const msg = message.toString();
            if (!msg.startsWith('$GPRP,')) return;

            const parts = msg.split(',');
            const [_, msgTag, gw, rssiString] = parts;
            
            // ALWAYS check against the input field's CURRENT value
            if (msgTag === this.fingerprintRefTag.value) {
                const rssi = parseFloat(rssiString);
                if (!isFinite(rssi)) return;

                const gwId = gw.toUpperCase();
                this.latestRssi[gwId] = rssi;
                this.updateRssiDisplay();

                if (this.isActivelyRecording) {
                    if (!this.recordingHistory[gwId]) this.recordingHistory[gwId] = [];
                    this.recordingHistory[gwId].push(rssi);

                    // Update status with live counts for each gateway
                    const counts = Object.entries(this.recordingHistory)
                        .map(([gateway, readings]) => `${gateway.slice(-4)}: ${readings.length}`)
                        .join(', ');
                    this.fingerprintStatusEl.textContent = `Recording... ${counts}`;

                } else if (!this.selectedFingerprintCoords) {
                    this.fingerprintStatusEl.textContent = `Select a point to record`;
                }
            }
        });

        this.mqttClient.on('error', (err) => { this.fingerprintStatusEl.textContent = `Connection error: ${err.message}`; });
        this.mqttClient.on('close', () => { if(this.isFingerprinting) this.fingerprintStatusEl.textContent = 'Connection closed.'; });
    }
    
    disconnectMqtt() {
        if (this.mqttClient) {
            this.mqttClient.end(); this.mqttClient = null;
            this.fingerprintStatusEl.textContent = 'Disconnected.';
            console.log("MQTT client disconnected.");
        }
    }

	updateRssiDisplay() {
		this.rssiReadingsEl.innerHTML = '';
		const allGateways = new Set([...Object.keys(this.latestRssi), ...Object.keys(this.gateways.reduce((acc, gw) => ({...acc, [gw.mac]:-120}), {}))]);
		const sortedGateways = Array.from(allGateways).sort();


		if (sortedGateways.length === 0) {
			 this.rssiReadingsEl.innerHTML = `<p class="e-rssi-no-gateways">No gateways configured.</p>`;
			 return;
		}

		for (const gw of sortedGateways) {
			const rssi = this.latestRssi[gw] || -100;
			const percentage = Math.max(0, Math.min(100, (rssi + 100)));

			const readingEl = this.createElement('div', ['e-rssi-reading']);
			readingEl.innerHTML = `
				<div class="e-rssi-reading-info">
					<span class="e-rssi-reading-mac">${gw}</span>
					<span class="e-rssi-reading-value">${rssi === -100 ? 'N/A' : rssi.toFixed(1) + ' dBm'}</span>
				</div>
				<div class="e-rssi-bar-bg">
					<div class="e-rssi-bar" style="width: ${percentage}%"></div>
				</div>
			`;
			this.rssiReadingsEl.appendChild(readingEl);
		}
	}
    
    recordFingerprint() {
        if (!this.selectedFingerprintCoords) {
            this.showAlert("Please select a point on the map first.");
            return;
        }

        // Toggle the recording state
        this.isActivelyRecording = !this.isActivelyRecording;

        if (this.isActivelyRecording) {
            // --- Start recording ---
            this.recordingHistory = {};
            this.recordFingerprintBtn.textContent = 'Stop';
            // Optional: Change button style to indicate it's active
			
            this.recordFingerprintBtn.classList.remove('e-planner-button-success');
            this.recordFingerprintBtn.classList.add('e-planner-button-danger');
            this.fingerprintStatusEl.textContent = `Recording... Click "Stop" when done.`;
        } else {
            // --- Stop recording and finalize ---
            this._finalizeAndStoreFingerprint();
            this.recordFingerprintBtn.textContent = 'Record';
            this.recordFingerprintBtn.classList.remove('e-planner-button-primary');
            this.recordFingerprintBtn.classList.add('e-planner-button-success');
        }
    }
    
    _finalizeAndStoreFingerprint() {
        this.isActivelyRecording = false; // Ensure recording is stopped

        const smoothedReadings = {};
        // Set a minimum number of readings required to be included in the fingerprint.
        // This prevents EMA calculation on too few data points.
        const MIN_READINGS_FOR_EMA = 3; 

        for (const gwId in this.recordingHistory) {
            const readings = this.recordingHistory[gwId];
            // Process any gateway that has a sufficient number of readings
            if (readings && readings.length >= MIN_READINGS_FOR_EMA) {
                smoothedReadings[gwId] = this._calculateEMA(readings);
            }
        }

        const gatewayCount = Object.keys(smoothedReadings).length;
        if (gatewayCount < 2) {
            this.showAlert(`Recording failed. Received sufficient readings (${MIN_READINGS_FOR_EMA}+) from only ${gatewayCount} gateways. At least 2 are required.`);
            this.fingerprintStatusEl.textContent = `Recording failed. Try moving to a better location.`;
            this.redrawCanvas();
            return;
        }

        const fingerprint = {
            location: this.selectedFingerprintCoords,
            readings: smoothedReadings,
        };
        
        this.fingerprints.push(fingerprint);
        console.log(`Fingerprint saved with data from ${gatewayCount} gateways:`, fingerprint);

        // --- Reset UI for the next recording ---
        this.fingerprintCountEl.textContent = `${this.fingerprints.length} recorded`;
        this.exportFingerprintsBtn.disabled = false;
        this.saveFingerprintsBtn.disabled = false;
        this.fingerprintStatusEl.textContent = `Fingerprint saved! Select a new point.`;
        
        this.selectedFingerprintCoords = null;
        this.selectedPointEl.textContent = 'Selected: None';
        this.recordFingerprintBtn.disabled = true;
        this.redrawCanvas();
    }

	deleteSelectedFingerprint() {
		if (!this.selectedFingerprintForDeletion) {
			return;
		}

		this.fingerprints = this.fingerprints.filter(fp => fp !== this.selectedFingerprintForDeletion);

		console.log("Fingerprint deleted:", this.selectedFingerprintForDeletion);

		this.selectedFingerprintForDeletion = null;
		this.deleteFingerprintBtn.disabled = true;
		this.fingerprintStatusEl.textContent = 'Fingerprint deleted. Select a point to record.';

		this.fingerprintCountEl.textContent = `${this.fingerprints.length} recorded`;
		this.exportFingerprintsBtn.disabled = this.fingerprints.length === 0;
		this.saveFingerprintsBtn.disabled = this.fingerprints.length === 0;
		this.redrawCanvas();
	}
	
    exportFingerprints() {
        if(this.fingerprints.length === 0) {
            this.showAlert("No fingerprints have been recorded yet.");
            return;
        }
		
		const fingerprintsInMeters = this.fingerprints.map(fp => {
			const locationInMeters = this.normToMeters(fp.location);
			return {
				location: {
					x: +locationInMeters.x.toFixed(3),
					y: +locationInMeters.y.toFixed(3)
				},
				readings: fp.readings
			};
		});
		
        const data = {
            timestamp: new Date().toISOString(),
            fingerprints: fingerprintsInMeters,
            map: this.image.src,
            tagId: this.options.mqtt.tagId
        };
        
        const dataStr = JSON.stringify(data, null, 2);
        const dataBlob = new Blob([dataStr], { type: 'application/json' });
        const url = URL.createObjectURL(dataBlob);
        
        const a = document.createElement('a');
        a.href = url; a.download = 'fingerprints.json';
        document.body.appendChild(a); a.click();
        document.body.removeChild(a); URL.revokeObjectURL(url);
    }

    // --- Mouse Event Handlers ---
    handleMouseDown(e) {
      const rect = this.canvas.getBoundingClientRect();
      const mouseX = e.clientX - rect.left;
      const mouseY = e.clientY - rect.top;

      if (this.isAddingGeofence) {
        if (!this.pendingGeofence) return;

        const g = this.pendingGeofence;
        
        if (g.nodes.length >= 3) {
          const firstNodeInScreenCoords = this.screenFromNorm(g.nodes[0]);
          // FIX: Use this.mousePosition to check distance for closing the shape, respecting the snap.
          const distanceToFirstNode = Math.hypot(this.mousePosition.x - firstNodeInScreenCoords.x, this.mousePosition.y - firstNodeInScreenCoords.y);
          
          if (distanceToFirstNode <= this.HIT_RADIUS_PX) {
            this.finishGeofence();
            return;
          }
        }
        
        // FIX: Use this.mousePosition to create the new node, ensuring it's placed at the snapped location.
        const newNodeCoords = this.screenToNorm(this.mousePosition.x, this.mousePosition.y);
        g.nodes.push(newNodeCoords);
        this.redrawCanvas();
        return;
      }

      if (this.isAddingGateway || this.isMeasuring || this.isFingerprinting) return;
      
      this.isDragging = false;
      this.dragStart   = { x: e.clientX - this.panX, y: e.clientY - this.panY };
      const previouslySelected = this.selectedGeofence || null;
      this.selectedPseudoNode = null; this.selectedNode = null; this.selectedGateway = null;
      if (previouslySelected) {
        const nodeRadius   = 12;
		const pseudoRadius = 12;

		const hitPseudo = this._hitPseudoAt(previouslySelected, mouseX, mouseY, pseudoRadius);
		if (hitPseudo) {
		  this.selectedGeofence   = previouslySelected;
		  this.selectedPseudoNode = hitPseudo;
		  this.isDraggingNode     = true;
		  this.isDraggingGateway  = false;
		  this.canvas.classList.add('grabbing');
		  this.removeNodeBtn.classList.add('hidden');
		  this.dragCapture = { type: 'geofence', ref: previouslySelected };
		  this.redrawCanvas(); this.updateButtonVisibility();
		  return;
		}

		const hitNode = this._hitNodeAt(previouslySelected, mouseX, mouseY, nodeRadius);
		if (hitNode) {
		  this.selectedGeofence   = previouslySelected;
		  this.selectedNode       = hitNode;
		  this.isDraggingNode     = true;
		  this.isDraggingGateway  = false;
		  this.canvas.classList.add('grabbing');
		  this.removeNodeBtn.classList.remove('hidden');
		  this.dragCapture = { type: 'geofence', ref: previouslySelected };
		  this.redrawCanvas(); this.updateButtonVisibility();
		  return;
		}

		if (this.pointInGeofence(mouseX, mouseY, previouslySelected)) {
		  this.selectedGeofence   = previouslySelected;
		  this.isDraggingNode     = false;
		  this.isDraggingGateway  = false;
		  this.dragCapture        = { type: 'geofence', ref: previouslySelected };
		  this.redrawCanvas(); this.updateButtonVisibility();
		  return;
		}
      }

      const clickedGateway = this.gateways.find(gw => {
        const gx = gw.x * this.originalImageDimensions.width * this.zoomFactor + this.panX;
        const gy = gw.y * this.originalImageDimensions.height * this.zoomFactor + this.panY;
        return Math.hypot(mouseX - gx, mouseY - gy) < 15;
      });
      if (clickedGateway) {
        this.dragInitialPosition = { x: clickedGateway.x, y: clickedGateway.y };
        this.selectedGateway = clickedGateway; this.isDraggingGateway = true;
        this.canvas.classList.add('grabbing'); this.redrawCanvas(); this.updateButtonVisibility(); return;
      }

      this.selectedGeofence = null; this.selectedGateway = null; this.dragCapture = null;
      this.canvas.classList.add('grabbing'); this.redrawCanvas(); this.updateButtonVisibility();
   
    }

    async handleMouseUp() {
		if (this.isDraggingGateway && this.selectedGateway) {
          const hasMoved = this.dragInitialPosition && 
                         (this.selectedGateway.x !== this.dragInitialPosition.x || 
                          this.selectedGateway.y !== this.dragInitialPosition.y);

          if (hasMoved) {
              this.fingerprintWarningModal.classList.remove('hidden');
          } else {
              this.isDraggingGateway = false;
              this.dragStart = null;
              this.canvas.classList.remove('grabbing');
          }
          return; 
      }

      if (this.isDraggingNode && this.dragCapture?.type === 'geofence') {
        const geofenceToUpdate = this.dragCapture.ref;
        try {
            if (this.selectedPseudoNode && geofenceToUpdate) {
                const newNode = { x: this.selectedPseudoNode.x, y: this.selectedPseudoNode.y };
                geofenceToUpdate.nodes.splice(this.selectedPseudoNode.insertIndex, 0, newNode);
                this.selectedNode = newNode; 
                this.selectedPseudoNode = null;
            }
            
            if (geofenceToUpdate?.id) {
                await this.updateGeofenceAPI(geofenceToUpdate);
            }
        } catch (err) {
            console.error('Failed to persist geofence after drag:', err);
            this.showAlert('Failed to save geofence position: ' + err.message);
        } finally {
            this.isDraggingNode = false;
            this.dragStart = null;
            this.dragCapture = null;
            this.canvas.classList.remove('grabbing');
            this.updateButtonVisibility(); 
            this.redrawCanvas();
        }
        return;
      }
      
      this.dragStart = null; 
      this.isDraggingGateway = false; 
      this.isDraggingNode = false;
      this.dragCapture = null;
      this.canvas.classList.remove('grabbing');
      this.updateButtonVisibility(); 
      this.redrawCanvas();
    }

    handleMouseMove(e) {
        const rect = this.canvas.getBoundingClientRect();
        const mouseX = e.clientX - rect.left;
        const mouseY = e.clientY - rect.top;

        this.mousePosition = { x: mouseX, y: mouseY };
		
		
		if (this.dragStart) {
            this.isDragging = true;
        }
		if (this.isMeasuring && this.measureStartPoint) {
			this.updateMeasureTooltip();
		}

        if (this.isAddingGeofence && this.geofences.length > 0 && this.geofences[this.geofences.length - 1].nodes.length > 0) {
            let snappedPosition = null;
            const allNodesAndGateways = [...this.geofences.flatMap(p => p.nodes), ...this.gateways];
            let minDistance = this.SNAP_DISTANCE + 1;

            allNodesAndGateways.forEach(item => {
                const itemPixelX = item.x * this.originalImageDimensions.width * this.zoomFactor + this.panX;
                const itemPixelY = item.y * this.originalImageDimensions.height * this.zoomFactor + this.panY;
                const distance = Math.sqrt(Math.pow(mouseX - itemPixelX, 2) + Math.pow(mouseY - itemPixelY, 2));

                if (distance < minDistance) {
                    minDistance = distance;
                    snappedPosition = { x: itemPixelX, y: itemPixelY };
                }
            });
            if (snappedPosition) {
                this.mousePosition = snappedPosition;
            }
            this.redrawCanvas();
        }

        if (this.isMeasuring && this.measureStartPoint) {
            this.redrawCanvas();
        }

        if (this.isDraggingNode && this.selectedNode) {
            const newX = (mouseX - this.panX) / (this.originalImageDimensions.width * this.zoomFactor);
            const newY = (mouseY - this.panY) / (this.originalImageDimensions.height * this.zoomFactor);

            let snapped = false;
            const allNodesAndGateways = [...this.geofences.flatMap(p => p.nodes), ...this.gateways];
            for (const other of allNodesAndGateways) {
                if (other === this.selectedNode) continue;
                const otherPixelX = other.x * this.originalImageDimensions.width * this.zoomFactor + this.panX;
                const otherPixelY = other.y * this.originalImageDimensions.height * this.zoomFactor + this.panY;
                const dist = Math.sqrt(Math.pow(mouseX - otherPixelX, 2) + Math.pow(mouseY - otherPixelY, 2));

                if (dist < this.SNAP_DISTANCE) {
                    this.selectedNode.x = other.x;
                    this.selectedNode.y = other.y;
                    snapped = true;
                    break;
                }
            }

            if (!snapped) {
                this.selectedNode.x = newX;
                this.selectedNode.y = newY;
            }
            this.redrawCanvas();
        } else if (this.isDraggingNode && this.selectedPseudoNode) {
            const newX = (mouseX - this.panX) / (this.originalImageDimensions.width * this.zoomFactor);
            const newY = (mouseY - this.panY) / (this.originalImageDimensions.height * this.zoomFactor);
            this.selectedPseudoNode.x = newX;
            this.selectedPseudoNode.y = newY;
            this.redrawCanvas();
        } else if (this.isDraggingGateway) {
            const newX = (mouseX - this.panX) / (this.originalImageDimensions.width * this.zoomFactor);
            const newY = (mouseY - this.panY) / (this.originalImageDimensions.height * this.zoomFactor);
            this.selectedGateway.x = newX;
            this.selectedGateway.y = newY;
            this.redrawCanvas();
        } else if (this.dragStart) {
            this.panX = e.clientX - this.dragStart.x;
            this.panY = e.clientY - this.dragStart.y;
            this.redrawCanvas();
        }
    }

    handleMouseClick(e) {
        if (this.isDragging) {
            this.isDragging = false;
            return;
        }

        const rect = this.canvas.getBoundingClientRect();
        const mouseX = e.clientX - rect.left;
        const mouseY = e.clientY - rect.top;

        if (this.isFingerprinting) {
            const CLICK_RADIUS = 8;
            let clickedFingerprint = null;
            for (const fp of this.fingerprints) {
                const pos = this.screenFromNorm(fp.location);
                const distance = Math.hypot(mouseX - pos.x, mouseY - pos.y);
                if (distance < CLICK_RADIUS) {
                    clickedFingerprint = fp;
                    break;
                }
            }
            if (clickedFingerprint) {
                this.selectedFingerprintForDeletion = clickedFingerprint;
                this.selectedFingerprintCoords = null;
                this.selectedPointEl.textContent = 'Selected for deletion';
                this.deleteFingerprintBtn.disabled = false;
                this.recordFingerprintBtn.disabled = true;
                this.fingerprintStatusEl.textContent = `Fingerprint selected. Press delete or click elsewhere.`;
                this.redrawCanvas();
                return;
            }
            this.selectedFingerprintForDeletion = null;
            this.deleteFingerprintBtn.disabled = true;
            this.selectedFingerprintCoords = this.screenToNorm(mouseX, mouseY);
            this.selectedPointEl.textContent = `(${this.selectedFingerprintCoords.x.toFixed(3)}, ${this.selectedFingerprintCoords.y.toFixed(3)})`;
            this.recordFingerprintBtn.disabled = false;
            this.fingerprintStatusEl.textContent = `Ready to record at selected point.`;
            this.redrawCanvas();
            return;
        }

        if (this.isMeasuring) {
            if (this.basePixelsPerMeter <= 0) {
              this.showAlert('Please set the scale / calibrate before measuring.');
              return;
            }
            let pt = { x: mouseX, y: mouseY };
            let snapped = null;
            let minDistance = this.SNAP_DISTANCE + 1;
            const candidates = [...this.geofences.flatMap(p => p.nodes), ...this.gateways];
            for (const item of candidates) {
              const itemX = item.x * this.originalImageDimensions.width  * this.zoomFactor + this.panX;
              const itemY = item.y * this.originalImageDimensions.height * this.zoomFactor + this.panY;
              const d = Math.hypot(mouseX - itemX, mouseY - itemY);
              if (d < minDistance) { minDistance = d; snapped = { x: itemX, y: itemY }; }
            }
            if (snapped) pt = snapped;

            if (!this.measureStartPoint) {
              this.measureStartPoint = pt;
              this.mousePosition = pt;
              this.updateMeasureTooltip();
              this.redrawCanvas();
            } else {
              const px = Math.hypot(pt.x - this.measureStartPoint.x, pt.y - this.measureStartPoint.y);
              const meters = (px / this.zoomFactor) / this.basePixelsPerMeter;
              this.canvas.dispatchEvent(new CustomEvent('echelonic:measured', {
                detail: { meters, pixels: px, start: this.measureStartPoint, end: pt }
              }));
              console.log(`Measured: ${meters.toFixed(2)} m`);
              this.measureStartPoint = null;
			  this.measureTooltip.classList.add('hidden');
              this._enterMode(null);
            }
            return;
        }
        
        if (this.isAddingGateway) {
            const n = this.screenToNorm(mouseX, mouseY);
            this.pendingGatewayNorm = n;
            this.showGatewayModal();
            return;
        }

        const previouslySelectedGeofence = this.selectedGeofence;

        if (previouslySelectedGeofence && !this.isAddingGeofence) {
            const hitNode = this._hitNodeAt(previouslySelectedGeofence, mouseX, mouseY, 10);
            if (hitNode) {
                this.selectedNode = hitNode;
                this.selectedPseudoNode = null;
                this.redrawCanvas();
                this.updateButtonVisibility();
                return;
            }
        }
        
        this.selectedGateway = null;
        this.selectedGeofence = null;
        this.selectedNode = null;
        this.selectedPseudoNode = null;

        const clickedGateway = this.gateways.find(gw => {
            const gwPixelX = gw.x * this.originalImageDimensions.width  * this.zoomFactor + this.panX;
            const gwPixelY = gw.y * this.originalImageDimensions.height * this.zoomFactor + this.panY;
            return Math.hypot(mouseX - gwPixelX, mouseY - gwPixelY) < 15;
        });
        if (clickedGateway) {
            this.selectedGateway = clickedGateway;
            this.redrawCanvas();
            this.updateButtonVisibility();
            return;
        }

        for (let i = this.geofences.length - 1; i >= 0; i--) {
            const geofence = this.geofences[i];
            const clickedNode = this._hitNodeAt(geofence, mouseX, mouseY, 10);
            if (clickedNode && !this.isAddingGeofence) {
                this.selectedGeofence = geofence;
                this.selectedNode = clickedNode;
                this.bringGeofenceToFront(geofence);
                this.redrawCanvas();
                this.updateButtonVisibility();
                return;
            }
        }

        for (let i = this.geofences.length - 1; i >= 0; i--) {
             const geofence = this.geofences[i];
             if (this.pointInGeofence(mouseX, mouseY, geofence) && !this.isAddingGeofence) {
                this.selectedGeofence = geofence;
                this.bringGeofenceToFront(geofence);
                this.redrawCanvas();
                this.updateButtonVisibility();
                return;
             }
        }
        
        this.redrawCanvas();
        this.updateButtonVisibility();
    }

    // --- Modal & State Management ---
    async saveGateway() {
        const macAddress = this.macAddressInput.value.trim(), gatewayName = this.gatewayNameInput.value.trim() || null;
        if (!macAddress) { this.showAlert("MAC Address is required."); return; }
        const finish = () => { this.hideGatewayModal(); this.redrawCanvas(); this.updateButtonVisibility(); };
        if (this.selectedGateway) {
            Object.assign(this.selectedGateway, { mac: macAddress, name: gatewayName});
            try {
                if (this.selectedGateway.id) await this.updateGatewayAPI(this.selectedGateway);
            } catch (e) { console.error("Failed to update gateway:", e); this.showAlert("Failed to update gateway: " + e.message);
            } finally { this.selectedGateway = null; finish(); }
        } else {
            const newGateway = { ...this.pendingGatewayNorm, mac: macAddress, name: gatewayName};
            try {
                const res = await this.createGatewayAPI(newGateway);
                newGateway.id = res.id;
                this.gateways.push(newGateway);
            } catch (e) { console.error("Failed to create gateway:", e); this.showAlert("Failed to create gateway: " + e.message);
            } finally { finish(); }
        }
    }
    cancelModal() {
        this.hideGatewayModal();
        this.selectedGateway = null;
        this.redrawCanvas();
        this.updateButtonVisibility();
    }
    showGatewayModal(gateway = null) {
        if (gateway) {
            this.gatewayModalTitle.textContent = 'Edit Gateway';
            this.macAddressInput.value = gateway.mac || ''; this.gatewayNameInput.value = gateway.name || ''; 
        } else {
            this.gatewayModalTitle.textContent = 'Add Gateway';
            this.macAddressInput.value = ''; this.gatewayNameInput.value = ''; 
        }
        this.gatewayModal.classList.remove('hidden'); this.macAddressInput.focus();
    }
    hideGatewayModal() {
        this.gatewayModal.classList.add('hidden'); this._enterMode(null);
    }
    async saveGeofence() {
        const name = this.geofenceNameInput.value.trim() || `Geofence ${this.geofenceIdCounter}`;
        const isActive = this.geofenceModal.querySelector('input[name="geofenceStatus"]:checked').value === 'active';
        
        const target = this.selectedGeofence || this.pendingGeofence;
        if (!target) {
            this.showAlert('No geofence to save.');
            return;
        }
        
        Object.assign(target, { name, is_active: isActive ? 1 : 0 });
        
        try {
            if (this.selectedGeofence && target.id) { // Updating an existing geofence
                await this.updateGeofenceAPI(target);
            } else if (this.pendingGeofence) { // Saving a new geofence
                const res = await this.createGeofenceAPI(target);
                target.id = res.id ?? this.geofenceIdCounter++;
                this.geofences.push(target); // Add to the main array NOW
            }
        } catch (e) {
            console.error('Failed to save geofence:', e);
            this.showAlert('Failed to save geofence: ' + e.message);
        } finally {
            this._enterMode(null); // This will clear pendingGeofence and selectedGeofence
            this.hideGeofenceModal();
        }
    }
    cancelGeofenceModal() {
        this.hideGeofenceModal();
        this.selectedGeofence = null;
        this.redrawCanvas();
        this.updateButtonVisibility();
    }
    showGeofenceModal(geofence = null) {
        if (geofence) {
            this.geofenceModalTitle.textContent = 'Edit Geofence'; this.geofenceNameInput.value = geofence.name || '';
            this.geofenceModal.querySelector(`input[value="${geofence.is_active ? 'active' : 'inactive'}"]`).checked = true;
        } else {
            this.geofenceModalTitle.textContent = 'Add Geofence'; this.geofenceNameInput.value = '';
            this.geofenceModal.querySelector('input[value="active"]').checked = true;
        }
        this.geofenceModal.classList.remove('hidden'); this.geofenceNameInput.focus();
    }
   hideGeofenceModal() {
        this.geofenceModal.classList.add('hidden');
        if (!this.isAddingGeofence) {
            this.selectedGeofence = null;
        }
        if (this.isAddingGeofence) {
            this._enterMode(null);
        }
        this.redrawCanvas();
        this.updateButtonVisibility();
    }

    // --- Zoom & Pan ---
	zoomBy(factor, anchorX, anchorY) {
		if (!this.image) return;
		const oldZoom = this.zoomFactor, newZoom = oldZoom * factor;
		const worldX = (anchorX - this.panX) / oldZoom, worldY = (anchorY - this.panY) / oldZoom;
		this.zoomFactor = newZoom;
		this.panX = anchorX - worldX * newZoom; this.panY = anchorY - worldY * newZoom;
		this.redrawCanvas();
	}
    zoomIn() { const rect = this.canvas.getBoundingClientRect(); this.zoomBy(1.2, rect.width / 2, rect.height / 2); }
	zoomOut() { const rect = this.canvas.getBoundingClientRect(); this.zoomBy(1 / 1.2, rect.width / 2, rect.height / 2); }
    zoomToFit() {
        if (!this.image) return;
        const canvasWidth = this.canvas.offsetWidth, canvasHeight = this.canvas.offsetHeight;
        const imageAspectRatio = this.originalImageDimensions.width / this.originalImageDimensions.height;
        const canvasAspectRatio = canvasWidth / canvasHeight;
        this.zoomFactor = (imageAspectRatio > canvasAspectRatio) ? canvasWidth / this.originalImageDimensions.width : canvasHeight / this.originalImageDimensions.height;
        this.panX = (canvasWidth - this.originalImageDimensions.width * this.zoomFactor) / 2;
        this.panY = (canvasHeight - this.originalImageDimensions.height * this.zoomFactor) / 2;
        this.redrawCanvas();
    }

    // --- Main Drawing Loop ---
    redrawCanvas() {
        if (!this.image) return;
        const dpr = window.devicePixelRatio || 1;
        const rect = this.canvas.getBoundingClientRect();
        this.canvas.width = rect.width * dpr; this.canvas.height = rect.height * dpr;
        this.ctx.setTransform(1, 0, 0, 1, 0, 0); this.ctx.scale(dpr, dpr);
        this.ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
        this.ctx.save();
        this.ctx.translate(this.panX, this.panY);
        this.ctx.scale(this.zoomFactor, this.zoomFactor);
        this.ctx.drawImage(this.image, 0, 0);
        this.ctx.restore();
		this.drawMeterGrid();
        if (this.heatmapType !== 'none') this.drawHeatmap();
        if (this.basePixelsPerMeter > 0) this.drawScaleRectangle(this.basePixelsPerMeter * this.zoomFactor, rect);
        this.drawGeofences();
        this.drawGateways();
        this.drawFingerprints();
        this.drawTapeMeasure();
    }

	// --- Modal & State Management ---
    showAlert(message, title = 'Notification') {
        this.alertModalTitle.textContent = title;
        this.alertModalMessage.textContent = message;
        this.alertModal.classList.remove('hidden');
    }
	
    // --- Drawing Sub-methods ---
	drawMeterGrid() {
        if (!this.grid?.enabled || this.basePixelsPerMeter <= 0) return;
        const ctx = this.ctx, W = this.canvas.width, H = this.canvas.height, zoom = this.zoomFactor || 1, ppm = this.basePixelsPerMeter;
        const panX = this.getPanX(), panY = this.getPanY();
        const screenToWorldMeters = (sx, sy) => ({ mx: ((sx - panX) / zoom) / ppm, my: ((sy - panY) / zoom) / ppm });
        const tl = screenToWorldMeters(0, 0), br = screenToWorldMeters(W, H);
        const minX = Math.min(tl.mx, br.mx), maxX = Math.max(tl.mx, br.mx);
        const minY = Math.min(tl.my, br.my), maxY = Math.max(tl.my, br.my);
        const stepM = this.grid.spacingMeters, majorN = Math.max(1, this.grid.majorEvery|0);
        const toScreenX = (mx) => (mx * ppm) * zoom + panX, toScreenY = (my) => (my * ppm) * zoom + panY;
        ctx.save(); ctx.lineWidth = this.grid.lineWidth;
        const crisp = (v) => Math.round(v) + 0.5;
        for (let i = Math.floor(minX / stepM); i <= Math.ceil(maxX / stepM); i++) {
            const isMajor = (i % majorN) === 0, sx = crisp(toScreenX(i * stepM));
            ctx.beginPath(); ctx.strokeStyle = isMajor ? "rgba(0,0,0,0.28)" : "rgba(0,0,0,0.12)";
            ctx.moveTo(sx, 0); ctx.lineTo(sx, H); ctx.stroke();
            if (this.grid.showLabels && isMajor) {
                ctx.font = "600 12px Inter, system-ui, sans-serif"; ctx.fillStyle = ctx.strokeStyle;
                ctx.textBaseline = 'top'; ctx.fillText(`${i} m`, sx + 4, 4);
            }
        }
        for (let j = Math.floor(minY / stepM); j <= Math.ceil(maxY / stepM); j++) {
            const isMajor = (j % majorN) === 0, sy = crisp(toScreenY(j * stepM));
            ctx.beginPath(); ctx.strokeStyle = isMajor ? "rgba(0,0,0,0.28)" : "rgba(0,0,0,0.12)";
            ctx.moveTo(0, sy); ctx.lineTo(W, sy); ctx.stroke();
            if (this.grid.showLabels && isMajor) {
                ctx.font = "600 12px Inter, system-ui, sans-serif"; ctx.fillStyle = ctx.strokeStyle;
                ctx.textBaseline = 'alphabetic'; ctx.fillText(`${j} m`, 4, sy - 4);
            }
        }
        ctx.restore();
    }
    drawTapeMeasure() {
		if (this.isMeasuring && this.measureStartPoint) {
			const startX = this.measureStartPoint.x, startY = this.measureStartPoint.y;
			const endX = this.mousePosition.x, endY = this.mousePosition.y;
			this.ctx.beginPath(); this.ctx.moveTo(startX, startY); this.ctx.lineTo(endX, endY);
			this.ctx.strokeStyle = 'rgba(0, 0, 0, 0.7)'; this.ctx.lineWidth = 3; this.ctx.setLineDash([5, 5]); this.ctx.stroke();
			this.ctx.setLineDash([]); this.ctx.beginPath(); this.ctx.arc(startX, startY, 5, 0, Math.PI * 2);
			this.ctx.fillStyle = 'rgba(0, 0, 0, 1)'; this.ctx.fill(); this.ctx.beginPath();
			this.ctx.arc(endX, endY, 5, 0, Math.PI * 2); this.ctx.fillStyle = 'rgba(0, 0, 0, 1)'; this.ctx.fill();
			this.updateMeasureTooltip();
		}
	}
	updateMeasureTooltip() {
		if (!this.measureTooltip) return;
		if (!(this.isMeasuring && this.measureStartPoint)) { this.measureTooltip.classList.add('hidden'); return; }
		const startX = this.measureStartPoint.x, startY = this.measureStartPoint.y, endX = this.mousePosition.x, endY = this.mousePosition.y;
		const px = Math.hypot(endX - startX, endY - startY), m  = (px / this.zoomFactor) / this.basePixelsPerMeter;
		this.measureTooltip.textContent = `${m.toFixed(2)} m`;
		this.measureTooltip.style.left = `${endX}px`; this.measureTooltip.style.top  = `${endY}px`;
		this.measureTooltip.classList.remove('hidden');
    }
    drawGeofences() {
      // 1. Draw all saved, completed geofences
      this.geofences.forEach(geofence => {
        if (geofence.nodes.length < 2) return; // Don't draw incomplete saved geofences
        
        this.ctx.save(); this.ctx.translate(this.panX, this.panY); this.ctx.scale(this.zoomFactor, this.zoomFactor);
        this.ctx.beginPath();
        this.ctx.moveTo(geofence.nodes[0].x * this.originalImageDimensions.width, geofence.nodes[0].y * this.originalImageDimensions.height);
        for (let i = 1; i < geofence.nodes.length; i++) {
            this.ctx.lineTo(geofence.nodes[i].x * this.originalImageDimensions.width, geofence.nodes[i].y * this.originalImageDimensions.height);
        }
        this.ctx.closePath();
        
        const style = getComputedStyle(this.parentContainer);
        this.ctx.fillStyle = style.getPropertyValue('--e-geofence-fill-color');
        this.ctx.fill();
        if (this.selectedGeofence === geofence) {
            this.ctx.fillStyle = style.getPropertyValue('--e-geofence-selected-fill-color');
            this.ctx.fill();
        }
        this.ctx.strokeStyle = style.getPropertyValue('--e-primary-color');
        this.ctx.lineWidth = 4 / this.zoomFactor;
        this.ctx.stroke();
        this.ctx.restore();

        if (this.selectedGeofence === geofence) {
          // Draw handles only for selected completed geofence
          geofence.nodes.forEach((node) => {
            const screenPos = this.screenFromNorm(node);
            this.ctx.beginPath(); this.ctx.arc(screenPos.x, screenPos.y, 7, 0, Math.PI * 2);
            this.ctx.strokeStyle = style.getPropertyValue('--e-primary-color');
            this.ctx.fillStyle = (node === this.selectedNode) ? "#fff" : style.getPropertyValue('--e-primary-color');
            if (node === this.selectedNode) { this.ctx.lineWidth = 6; this.ctx.stroke(); } this.ctx.fill();
          });
          const pseudoNodes = this.updatePseudoNodes(geofence.nodes);
          pseudoNodes.forEach(p => {
            const screenPos = this.screenFromNorm(p);
            this.ctx.beginPath(); this.ctx.arc(screenPos.x, screenPos.y, 5, 0, Math.PI * 2);
            this.ctx.fillStyle = style.getPropertyValue('--e-primary-color'); this.ctx.fill();
          });
        }
      });

      // 2. Draw the pending geofence if it exists
      if (this.isAddingGeofence && this.pendingGeofence && this.pendingGeofence.nodes.length > 0) {
        const g = this.pendingGeofence;
        const style = getComputedStyle(this.parentContainer);
        
        this.ctx.save(); this.ctx.translate(this.panX, this.panY); this.ctx.scale(this.zoomFactor, this.zoomFactor);
        this.ctx.beginPath();
        this.ctx.moveTo(g.nodes[0].x * this.originalImageDimensions.width, g.nodes[0].y * this.originalImageDimensions.height);
        for (let i = 1; i < g.nodes.length; i++) {
            this.ctx.lineTo(g.nodes[i].x * this.originalImageDimensions.width, g.nodes[i].y * this.originalImageDimensions.height);
        }
        this.ctx.strokeStyle = style.getPropertyValue('--e-primary-color');
        this.ctx.lineWidth = 4 / this.zoomFactor;
        this.ctx.stroke();
        this.ctx.restore();

        g.nodes.forEach((node) => {
            const screenPos = this.screenFromNorm(node);
            this.ctx.beginPath(); this.ctx.arc(screenPos.x, screenPos.y, 7, 0, Math.PI * 2);
            this.ctx.fillStyle = style.getPropertyValue('--e-primary-light-color'); this.ctx.fill();
        });

        const lastPos = this.screenFromNorm(g.nodes[g.nodes.length - 1]);
        this.ctx.beginPath(); this.ctx.moveTo(lastPos.x, lastPos.y);
        this.ctx.lineTo(this.mousePosition.x, this.mousePosition.y);
        this.ctx.strokeStyle = style.getPropertyValue('--e-primary-color'); this.ctx.lineWidth = 4; this.ctx.stroke();
      }
    }
    drawGateways() {
        const style = getComputedStyle(this.parentContainer);
        const fillColor = style.getPropertyValue('--e-inverted-color');
        const selectedFillColor = style.getPropertyValue('--e-inverted-color'); // Can be different if needed
        const strokeColor = style.getPropertyValue('--e-inverted-dark-color');

        this.gateways.forEach(gw => {
            const screenPos = this.screenFromNorm(gw);
            this.ctx.beginPath(); this.ctx.arc(screenPos.x, screenPos.y, 10, 0, Math.PI * 2);
            this.ctx.fillStyle = gw === this.selectedGateway ? selectedFillColor : fillColor;
            this.ctx.globalAlpha = gw === this.selectedGateway ? 1.0 : 0.8;
            this.ctx.fill(); 
            this.ctx.globalAlpha = 1.0;
            this.ctx.strokeStyle = strokeColor;
			this.ctx.lineWidth = 3; this.ctx.stroke();
        });
    }

	drawFingerprints() {
		this.fingerprints.forEach(fp => {
			const pos = this.screenFromNorm(fp.location);
			this.ctx.beginPath();
			this.ctx.arc(pos.x, pos.y, 6, 0, Math.PI * 2);

			if (fp === this.selectedFingerprintForDeletion) {
				this.ctx.fillStyle = 'rgba(220, 38, 38, 0.9)'; // Red for selected
			} else {
				this.ctx.fillStyle = 'rgba(37, 99, 235, 0.7)'; // Blue for normal
			}
			
			this.ctx.fill();
			this.ctx.strokeStyle = 'white';
			this.ctx.lineWidth = 2;
			this.ctx.stroke();
		});

		if (this.isFingerprinting && this.selectedFingerprintCoords) {
			const pos = this.screenFromNorm(this.selectedFingerprintCoords);
			this.ctx.beginPath();
			this.ctx.arc(pos.x, pos.y, 8, 0, Math.PI * 2);
			this.ctx.fillStyle = 'rgba(34, 197, 94, 0.9)';
			this.ctx.fill();
			this.ctx.strokeStyle = 'white';
			this.ctx.lineWidth = 2;
			this.ctx.stroke();
		}
	}
    drawScaleRectangle(pixelsPerMeter, canvasRect) {
        if (pixelsPerMeter === 0) return;
        const rectSize = pixelsPerMeter, rectX = canvasRect.width - rectSize - 20, rectY = canvasRect.height - rectSize - 20; 
        this.ctx.fillStyle = 'rgba(0, 0, 0, 0.4)'; this.ctx.fillRect(rectX, rectY, rectSize, rectSize);
        this.ctx.fillStyle = 'white'; this.ctx.font = 'bold 12px Inter';
        this.ctx.textAlign = 'center'; this.ctx.textBaseline = 'middle';
        if (rectSize > 25) this.ctx.fillText('1m²', rectX + rectSize / 2, rectY + rectSize / 2);
    }
    createHeatmapGradient() {
        const canvas = document.createElement('canvas'); canvas.height = 256;
        const ctx = canvas.getContext('2d'), grad = ctx.createLinearGradient(0, 0, 0, 256);
        grad.addColorStop(0.25, 'rgba(0, 0, 255, 0)'); grad.addColorStop(0.55, 'rgba(0, 255, 255, 0.7)');
        grad.addColorStop(0.85, 'yellow'); grad.addColorStop(1.0, 'red');
        ctx.fillStyle = grad; ctx.fillRect(0, 0, 1, 256);
        return ctx.getImageData(0, 0, 1, 256).data;
    }
    drawHeatmap() {
        if (this.gateways.length === 0 || this.basePixelsPerMeter === 0) return;
        if (!this.heatmapCanvas) {
            this.heatmapCanvas = document.createElement('canvas');
            this.heatmapGradient = this.createHeatmapGradient();
        }
        const canvasW = this.canvas.width, canvasH = this.canvas.height;
        if (this.heatmapCanvas.width !== canvasW || this.heatmapCanvas.height !== canvasH) {
            this.heatmapCanvas.width = canvasW; this.heatmapCanvas.height = canvasH;
        }
        const hCtx = this.heatmapCanvas.getContext('2d', { willReadFrequently: true });
        hCtx.clearRect(0, 0, canvasW, canvasH);
        const radiusMeters = ({ office: 10, residential: 15, industrial: 30, outdoors: 75 })[this.heatmapType] || 0;
        const radiusPixels = radiusMeters * this.basePixelsPerMeter * this.zoomFactor, intensity = 0.7;
        for (const gw of this.gateways) {
            const screenPos = this.screenFromNorm(gw);
            const grad = hCtx.createRadialGradient(screenPos.x, screenPos.y, 0, screenPos.x, screenPos.y, radiusPixels);
            grad.addColorStop(0, `rgba(0,0,0,${intensity})`); grad.addColorStop(1, 'rgba(0,0,0,0)');
            hCtx.fillStyle = grad; hCtx.beginPath(); hCtx.arc(screenPos.x, screenPos.y, radiusPixels, 0, Math.PI * 2); hCtx.fill();
        }
        const imgData = hCtx.getImageData(0, 0, canvasW, canvasH), data = imgData.data, gradData = this.heatmapGradient;
        for (let i = 0; i < data.length; i += 4) {
            const a = data[i + 3];
            if (a > 0) {
                const idx = a * 4;
                data[i] = gradData[idx]; data[i + 1] = gradData[idx + 1]; data[i + 2] = gradData[idx + 2]; data[i + 3] = gradData[idx + 3];
            }
        }
        hCtx.putImageData(imgData, 0, 0); this.ctx.drawImage(this.heatmapCanvas, 0, 0);
    }
    
    // --- UI Update & Helpers ---
    updatePseudoNodes(nodes) {
        if (nodes.length < 2) return [];
        const pseudoNodes = [];
        for (let i = 0; i < nodes.length; i++) {
            const node1 = nodes[i], node2 = nodes[(i + 1) % nodes.length];
            pseudoNodes.push({ x: (node1.x + node2.x) / 2, y: (node1.y + node2.y) / 2, parent1: node1, parent2: node2, insertIndex: i + 1 });
        }
        return pseudoNodes;
    }
    pointInGeofence(screenX, screenY, geofence) {
      if (!geofence?.nodes?.length) return false;
      const pts = geofence.nodes.map(n => this.screenFromNorm(n));
      let inside = false;
      for (let i = 0, j = pts.length - 1; i < pts.length; j = i++) {
        const xi = pts[i].x, yi = pts[i].y, xj = pts[j].x, yj = pts[j].y;
        if (((yi > screenY) !== (yj > screenY)) && (screenX < (xj - xi) * (screenY - yi) / (yj - yi + 1e-5) + xi)) inside = !inside;
      }
      return inside;
    }
    bringGeofenceToFront(geofence) {
      if (!this.geofences || !geofence) return;
      const idx = this.geofences.indexOf(geofence);
      if (idx > -1) { this.geofences.splice(idx, 1); this.geofences.push(geofence); }
    }
    updateButtonVisibility() {
        const isInteracting = this.isAddingGateway || this.isAddingGeofence || this.isMeasuring || this.isFingerprinting;
        this.gatewayControls.classList.toggle('hidden', isInteracting || !this.image);
        this.fingerprintPanel.classList.toggle('hidden', !this.isFingerprinting);

        let showSelectionControls = false;
        this.removeGatewayBtn.classList.add('hidden'); this.editGatewayBtn.classList.add('hidden');
        this.editGeofenceBtn.classList.add('hidden'); this.removeGeofenceBtn.classList.add('hidden');
        this.removeNodeBtn.classList.add('hidden'); 
		this.finishGeofenceBtn.classList.add('hidden');
		this.cancelGeofenceBtn.classList.add('hidden');

        if (this.selectedGateway) {
            this.removeGatewayBtn.classList.remove('hidden'); this.editGatewayBtn.classList.remove('hidden');
            showSelectionControls = true;
        } else if (this.isAddingGeofence) {
			this.finishGeofenceBtn.classList.remove('hidden');
			this.cancelGeofenceBtn.classList.remove('hidden');
            showSelectionControls = true;
        } else if (this.selectedGeofence) {
          this.editGeofenceBtn.classList.remove('hidden'); this.removeGeofenceBtn.classList.remove('hidden');
          this.removeNodeBtn.classList.toggle('hidden', !this.selectedNode);
          showSelectionControls = true;
        }
        this.selectionControls.classList.toggle('hidden', !showSelectionControls);
    }

    // ===== API INTEGRATION =====
    _reifyBufferedAPIData() {
        if (!this._apiBuffer) return;
        const ready = this.basePixelsPerMeter > 0 && this.originalImageDimensions.width > 0;
        if (!ready) return;
        if (Array.isArray(this._apiBuffer.gateways)) {
            this.gateways = this._apiBuffer.gateways.map(g => ({ ...this.metersToNorm({ x: +g.pos_x, y: +g.pos_y }), id: g.id, mac: g.mac_address, name: g.name, weight: g.weight, is_active: g.is_active }));
            this._apiBuffer.gateways = null;
        }
        if (Array.isArray(this._apiBuffer.geofences)) {
            this.geofences = this._apiBuffer.geofences.map(f => {
                let vertices = f.vertices;
                if (typeof vertices === 'string') { try { vertices = JSON.parse(vertices); } catch {} }
                const nodes = Array.isArray(vertices) ? vertices.map(([mx, my]) => this.metersToNorm({ x: +mx, y: +my })) : [];
                return { id: f.id, name: f.name, is_active: f.is_active, nodes };
            });
            this._apiBuffer.geofences = null;
        }
    }
    get baseUrl() { return (this.options.baseUrl || '').replace(/\/$/, ''); }
    get endpoints() { return this.options.endpoints || {}; }
    get authToken() { return this.options.authToken || null; }
    async apiFetch(path, init = {}) {
	  const url = this.baseUrl + path;
	  const headers = { ...(init.headers || {}), 'Content-Type': 'application/json' };
	  if (this.authToken) headers['Authorization'] = `Bearer ${this.authToken}`;
	  const res = await fetch(url, { ...init, headers });
	  if (!res.ok) {
		const bodyText = await res.text().catch(() => '');
		throw new Error(`API ${init.method || 'GET'} ${url} -> ${res.status}: ${bodyText}`);
	  }
	  const ct = res.headers.get('content-type') || '';
	  return ct.includes('application/json') ? res.json() : res.text();
	}
    normToMeters(norm) { return { x: (norm.x * this.originalImageDimensions.width) / this.basePixelsPerMeter, y: (norm.y * this.originalImageDimensions.height) / this.basePixelsPerMeter }; }
    metersToNorm(m) { return { x: (m.x * this.basePixelsPerMeter) / this.originalImageDimensions.width, y: (m.y * this.basePixelsPerMeter) / this.originalImageDimensions.height }; }
    async loadGatewaysFromAPI() {
	  try {
		const raw = await this.apiFetch(this.endpoints.listGateways, { method: 'GET' });
		const list = Array.isArray(raw) ? raw : (raw?.data || raw?.gateways || []);
		this.gateways = list.map(g => {
		  const n = this.metersToNorm({ x: +(g.pos_x ?? 0), y: +(g.pos_y ?? 0) });
		  return { id: g.id, mac: g.mac_address ?? '', name: g.name ?? null, weight: g.weight ?? null, x: n.x, y: n.y, is_active: g.is_active ?? 1 };
		});
		this.redrawCanvas();
	  } catch (err) { console.error('Failed to load gateways:', err); }
	}
    async createGatewayAPI(gw) { const m = this.normToMeters(gw); return this.apiFetch(this.endpoints.createGateway, { method: 'POST', body: JSON.stringify({ mac_address: gw.mac, name: gw.name, weight: gw.weight, pos_x: +m.x.toFixed(2), pos_y: +m.y.toFixed(2), is_active: 1 }) }); }
    async updateGatewayAPI(gw) { const m = this.normToMeters(gw); return this.apiFetch(this.endpoints.updateGateway.replace('{id}', gw.id), { method: 'PUT', body: JSON.stringify({ mac_address: gw.mac, name: gw.name, weight: gw.weight, pos_x: +m.x.toFixed(2), pos_y: +m.y.toFixed(2), is_active: gw.is_active ?? 1 }) }); }
    async deleteGatewayAPI(gw) { return this.apiFetch(this.endpoints.deleteGateway.replace('{id}', gw.id), { method: 'DELETE' }); }
    async loadGeofencesFromAPI() {
	  try {
		const raw = await this.apiFetch(this.endpoints.listGeofences, { method: 'GET' });
		const list = Array.isArray(raw) ? raw : (raw?.data || raw?.geofences || []);
		this.geofences = list.map(f => {
		  let vertices = f.vertices ?? [];
		  if (typeof vertices === 'string') try { vertices = JSON.parse(vertices); } catch {}
		  const nodes = (vertices || []).map(v => this.metersToNorm({ x: +(v[0] ?? 0), y: +(v[1] ?? 0)}));
		  return { id: f.id, name: f.name ?? null, is_active: f.is_active ?? 1, nodes };
		});
		this.redrawCanvas();
	  } catch (err) { console.error('Failed to load geofences:', err); }
	}
    async createGeofenceAPI(f) { const verts = f.nodes.map(n => { const m = this.normToMeters(n); return [ +m.x.toFixed(2), +m.y.toFixed(2) ]; }); return this.apiFetch(this.endpoints.createGeofence, { method: 'POST', body: JSON.stringify({ name: f.name, vertices: JSON.stringify(verts), is_active: f.is_active ?? 1 }) }); }
    async updateGeofenceAPI(f) { const verts = f.nodes.map(n => { const m = this.normToMeters(n); return [ +m.x.toFixed(2), +m.y.toFixed(2) ]; }); return this.apiFetch(this.endpoints.updateGeofence.replace('{id}', f.id), { method: 'PUT', body: JSON.stringify({ name: f.name, vertices: JSON.stringify(verts), is_active: f.is_active ?? 1 }) }); }
    async deleteGeofenceAPI(f) { return this.apiFetch(this.endpoints.deleteGeofence.replace('{id}', f.id), { method: 'DELETE' }); }
	async saveFingerprintsToServer() {
        if (this.fingerprints.length === 0) {
            this.showAlert("No fingerprints have been recorded to save.");
            return;
        }

        const fingerprintsInMeters = this.fingerprints.map(fp => {
            const locationInMeters = this.normToMeters(fp.location);
            return {
                location: {
                    x: +locationInMeters.x.toFixed(3),
                    y: +locationInMeters.y.toFixed(3)
                },
                readings: fp.readings
            };
        });

        const data = {
            timestamp: new Date().toISOString(),
            fingerprints: fingerprintsInMeters,
            map: this.image.src,
            tagId: this.options.mqtt.tagId
        };

        try {
            await this.updateFingerprintsAPI(data);
            this.showAlert('Fingerprints saved to server successfully!');
        } catch (err) {
            console.error('Failed to save fingerprints to server:', err);
            this.showAlert(`Error saving fingerprints: ${err.message}`);
        }
    }

    async updateFingerprintsAPI(data) {
        return this.apiFetch(this.endpoints.updateFingerprints, {
            method: 'POST',
            body: JSON.stringify(data)
        });
    }
}