/*! echelonic.map.webgl.js
 * Build: 2025-09-17
 * WebGL-only rewrite (no 2D canvas, no external libraries)
 * Description: This version has been refactored to remove all inline CSS.
 * Styling is now handled by a separate echelonic.map.css file.
 * The script uses CSS classes and CSS variables for theming and dynamic states.
 *
 * Public API (compatible names where possible):
 * new echelonicMap(options)
 * .addLayer(name, url) -> Returns Layer instance for chaining
 * .showLayer(name)
 * .hideActiveLayer()
 * .setOptions(newOptions)
 * .updateMapOverlay(config) // Note: Operates on the active layer
 * .fitOverlayToGeofences({ by, padding, align }, layer?)
 * .fitGeofencesInView()
 * .renderData(summary)
 * .analyzeTagData(tagUpdates)
 * .setFov(deg), .setPerspectiveEnabled(bool)
 * .zoomAt(x,y,factor), .rotateBy(rad, ax, ay), .setRotation(rad), .setTilt(rad)
 * .worldToScreen(x,y), .screenToWorld(x,y)
 */

/**
 * Manages the state and configuration of a single map layer.
 * Provides a fluent (chainable) interface for configuration.
 */
class Layer {
  constructor(name, url, mapInstance) {
    this.name = name;
    this.url = url;
    this._map = mapInstance; // Reference to the main map

    // Layer-specific properties
    this.img = null;
    this.texture = null;
    this.loaded = false;
    this.naturalW = 0;
    this.naturalH = 0;
    this.x = 0;
    this.y = 0;
    this.width = null;
    this.height = null;
    this.opacity = 1;
    this.lockAspect = true;
    this.vertexCount = 32 * 32 * 6; // Default from overlaySubdivisions
  }

  /** Sets the world position (top-left corner) of the layer. */
  setPosition(x, y) {
    this.x = x;
    this.y = y;
    this._map._draw();
    return this; // Enable chaining
  }

  /** Sets the world size of the layer. Aspect ratio is maintained if one dimension is null. */
  setSize({ width = null, height = null }) {
    this.width = width;
    this.height = height;
    this._map._recalcOverlaySizeIfNeeded(this); // Pass layer context
    this._map._draw();
    return this;
  }
  
  /** Sets the opacity of the layer. */
  setOpacity(opacity) {
    this.opacity = this._map._clamp(opacity, 0, 1);
    this._map._draw();
    return this;
  }

  /** Automatically scales and positions the layer to fit the geofences. */
  fitToGeofences(options = { by: 'width', padding: 1.0, align: 'top-left' }) {
    this._map.fitOverlayToGeofences(options, this); // Pass layer context
    this._map._draw();
    return this;
  }

  /** Makes this layer the currently visible one. */
  show() {
    this._map.showLayer(this.name);
    return this;
  }
}


class echelonicMap {
  constructor(options = {}) {
    this._initializeOptions(options);
    this._initializeState();
    this._setupCSSVariables();
    this._setupDOM();
    this._glInit();
    this.init();
  }

  // -----------------------------------------------------------------------------
  // Init & Options
  // -----------------------------------------------------------------------------
  _initializeOptions(options) {
    const defaults = {
		stabilizeOnRotate: true,
	  panAnimationMs: 200,
      autoCenter: true,
      enableZoom: true,
      enableRotate: true,
      enableTilt: true,
      enableTopDown: true,
      authToken: null,
      baseUrl: null,
      parentContainer: document.body,
      theme: '#0071e3',
      heatmap: true,
      tagMarkers: true,
      geofences: true,
      stats: true,
      totalText: 'Currently online',
      logo: true,
      updateInterval: 2000,
      // mapOverlay: null, // This is now handled by the Layer API

      // Camera
      tilt: 10,
      tiltStep: Math.PI / 18,
      maxTilt: Math.PI / 3,
      rotation: 0.78,
      zoom: 1,

      // Perspective
      perspective: true,
      autoFov: true,
      autoFovTargetRatio: 1.1,
      autoFovSmoothing: 0.1,
      fovDeg: 20,
      minFovDeg: 5,
      maxFovDeg: 100,
      tiltTouchSensitivity: 0.01,

      // Extrusion
      extrudeHeight: 3,

      // Roof transparency
      roofOpacity: 0.0,
      geofenceHeights: null,

      // Heatmap
      heatRadiusPx: 25,
      heatIntensity: 0.7,
      heatOpacity: 0.8,
      heatBlur: 5.0,
	  overlaySubdivisions: 32,
	  
	  // --- Marker Customization ---
      markerRadius: 10,
      markerFillColor: '#ffffff',
	  markerBorderThickness: 0.3, 
	  
	  trailsEnabled: true,
      trailLength: 50, // Number of historical points to store
      trailColor: null, // Can be any hex color
    };
    this.options = { ...defaults, ...options };
    if (this.options.updateInterval < 2000) this.options.updateInterval = 2000;
  }

  // _initiateAutoCenterAnimation() {
    // if (!this.options.autoCenter || this.isAnimatingPan) return;

    // const worldBounds = this._computeCombinedBounds();
    // if (!worldBounds || !isFinite(worldBounds.width) || worldBounds.width <= 0) return;

    // const corners = [
      // this.worldToScreen(worldBounds.minX, worldBounds.minY),
      // this.worldToScreen(worldBounds.maxX, worldBounds.minY),
      // this.worldToScreen(worldBounds.minX, worldBounds.maxY),
      // this.worldToScreen(worldBounds.maxX, worldBounds.maxY),
    // ];

    // const screenBounds = {
      // minX: Math.min(corners[0].x, corners[1].x, corners[2].x, corners[3].x),
      // minY: Math.min(corners[0].y, corners[1].y, corners[2].y, corners[3].y),
      // maxX: Math.max(corners[0].x, corners[1].x, corners[2].x, corners[3].x),
      // maxY: Math.max(corners[0].y, corners[1].y, corners[2].y, corners[3].y),
    // };
    // screenBounds.width = screenBounds.maxX - screenBounds.minX;
    // screenBounds.height = screenBounds.maxY - screenBounds.minY;

    // if ((screenBounds.width * 1.5) < this.canvas.width && (screenBounds.height * 1.5) < this.canvas.height) {
      // const worldCenter = { x: worldBounds.centerX, y: worldBounds.centerY };
      // const projectedCenter = this.worldToScreen(worldCenter.x, worldCenter.y);
      // const canvasCenter = { x: this.canvas.width / 2, y: this.canvas.height / 2 };
      // const dx = canvasCenter.x - projectedCenter.x;
      // const dy = canvasCenter.y - projectedCenter.y;

      // if (Math.hypot(dx, dy) < 1) return; // Already centered, no animation needed.

      // this.isAnimatingPan = true;
      // this.animationStart = performance.now();
      // this.panStart = { ...this.panOffset };
      // this.panEnd = { x: this.panOffset.x + dx, y: this.panOffset.y + dy };

      // Start the animation loop
      // this._animatePan();
    // }
  // }

  _animatePan() {
    if (!this.isAnimatingPan) return; // Stop if animation was cancelled

    const now = performance.now();
    const elapsed = now - this.animationStart;
    const duration = this.options.panAnimationMs;
    const progress = Math.min(elapsed / duration, 1.0);

    // Ease-out function for a nice slowing down effect
    const easedProgress = 1 - Math.pow(1 - progress, 3);

    this.panOffset.x = this._lerp(this.panStart.x, this.panEnd.x, easedProgress);
    this.panOffset.y = this._lerp(this.panStart.y, this.panEnd.y, easedProgress);

    this._draw();

    if (progress < 1.0) {
      requestAnimationFrame(() => this._animatePan());
    } else {
      this.isAnimatingPan = false;
    }
  }

  _initializeState() {
    this.panOffset = { x: 0, y: 0 };
    this.zoom = this.options.zoom || 1;
    this.rotation = this.options.rotation || 0;
    this.tilt = this._clampTilt(this.options.tilt);

    // Data
    this.geofences = [];
    this.tags = [];
    this.tagHeatmapData = {};
	this.tagTrails = {};

    // Layer Management
    this.layers = {};
    this.activeLayer = null;

    // NEW: Promise to ensure geofences are loaded before fitting operations
    this._geofencesReadyPromise = new Promise(resolve => {
      this._resolveGeofencesReady = resolve;
    });

    // GPU buffers
    this._buffers = {
      poly: null, polyColor: null, side: null, sideColor: null,
      marker: null, heatQuad: null, overlayQuad: null,
	  trail: null, trailColor: null,
    };

    // Per-frame dynamic arrays
    this._geom = {
      polyVerts: new Float32Array(0), polyCols:  new Float32Array(0),
      sideVerts: new Float32Array(0), sideCols:  new Float32Array(0),
      markers:   [], heats:     [],
    };

    // Input
    this.isPanning = false;
    this.isRotatingAndTilting = false;
	this.isAnimatingPan = false;
    this.animationStart = null;
    this.panStart = null;
    this.panEnd = null;
    this.zoomEndTimer = null;
    this.dragStart = null;
    this.lastPos = null;
	this.lastTouchPanPos = null;
    this.initialPinch = null;
    this.initialRotateAngle = null;
    this.initialTilt = null;
    this.initialMidY = null;
    this._gestureAnchorScreen = null;
    this._gestureAnchorWorld = null;

    // Misc
    this._heatGradientTex = null;
    this._heatFBO = null;
    this._heatIntensityTex = null;
    this._blurredIntensityTex = null;
    this._fullscreenQuadBuffer = null;
    this.isHistoryView = false;
  }

  _setupCSSVariables() {
    this.parent = typeof this.options.parentContainer === 'string'
      ? document.querySelector(this.options.parentContainer)
      : this.options.parentContainer;
    if (!this.parent) throw new Error('Parent container not found');
    
    this.themeHC = this.invertHexColor(this.options.theme, true);
    this.invertedTheme = this.invertHexColor(this.options.theme, false);
    this.invertedThemeHC = this.invertHexColor(this.invertedTheme, true);

    this.parent.style.setProperty('--echelonic-theme-color_1', this.options.theme);
    this.parent.style.setProperty('--echelonic-theme-color_2', this.themeHC);
    this.parent.style.setProperty('--echelonic-theme-color_3', this.invertedTheme);
    this.parent.style.setProperty('--echelonic-theme-color_4', this.invertedThemeHC);
  }

  _createGradientTexture(gl) {
    const canvas = document.createElement('canvas');
    const ctx = canvas.getContext('2d');
    canvas.width = 256;
    canvas.height = 1;

    const gradient = ctx.createLinearGradient(0, 0, 256, 0);
    gradient.addColorStop(0,    'rgba(0, 0, 255, 0)');
    gradient.addColorStop(0.25, '#00ffff');
    gradient.addColorStop(0.5,  '#00ff00');
    gradient.addColorStop(0.75, '#ffff00');
    gradient.addColorStop(1,    '#ff0000');

    ctx.fillStyle = gradient;
    ctx.fillRect(0, 0, 256, 1);

    const texture = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, texture);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, canvas);
    
    return texture;
  }

	// Add this helper method inside the echelonicMap class for cleaner code
  _handleTooltip = (e) => {
    // Only show tooltip if tag markers are enabled and not dragging
    if (!this.options.tagMarkers || this.isPanning || this.isRotatingAndTilting) {
      this.tooltipElement.style.display = 'none';
      return;
    }

    const rect = this.canvas.getBoundingClientRect();
    const mouseX = e.clientX - rect.left;
    const mouseY = e.clientY - rect.top;
    const radius = this.options.markerRadius;

    let foundTag = null;

    // Find the topmost tag under the cursor
    for (let i = this.tags.length - 1; i >= 0; i--) {
      const tag = this.tags[i];
      if (isNaN(tag.lon) || isNaN(tag.lat)) continue;
      
      const worldPos = this._geoToPixel(tag.lon, tag.lat);
      const screenPos = this.worldToScreen(worldPos.x, worldPos.y);
      
      const dx = mouseX - screenPos.x;
      const dy = mouseY - screenPos.y;
      
      if (Math.hypot(dx, dy) < radius) {
        foundTag = tag;
        break; // Found the topmost tag, no need to check others
      }
    }

    if (foundTag) {
      this.tooltipElement.style.display = 'block';
      this.tooltipElement.style.left = `${e.clientX + 15}px`; // Position relative to viewport
      this.tooltipElement.style.top = `${e.clientY + 15}px`;
		this.tooltipElement.textContent = foundTag.id;
    } else {
      this.tooltipElement.style.display = 'none';
    }
  }

  _setupDOM() {
    this.parent.classList.add('echelonic-map-container');

    // WebGL canvas
    this.canvas = document.createElement('canvas');
    this.canvas.className = 'echelonic-map__canvas hidden';
    this.parent.appendChild(this.canvas);

    // Message box
    this.messageBox = document.createElement('div');
    this.messageBox.className = 'echelonic-map__message-box';
    this.parent.appendChild(this.messageBox);

    // Stats container
    this.statsBox = document.createElement('div');
    this.statsBox.className = 'echelonic-map__stats-box';
    this.statsBox.classList.toggle('hidden', !this.options.stats);
    this.parent.appendChild(this.statsBox);

    // Online block
    this.statsBoxOnline = document.createElement('div');
    this.statsBoxOnline.className = 'stats-box__online';
    this.statsBox.appendChild(this.statsBoxOnline);

    // Geofence stats
    this.statsBoxGF = document.createElement('div');
    this.statsBoxGF.className = 'stats-box__geofences';
    this.statsBox.appendChild(this.statsBoxGF);

    // Controls
    this._injectControls();

    // Logo
    if (this.options.logo) {
      const a = document.createElement('a');
      a.className = 'echelonic-map__logo';
      a.href = 'https://echelonic.cc'; a.target = 'about:blank';
      a.innerHTML = `<svg height="12px" version="1.1" viewBox="0 0 131.07 22.706" xml:space="preserve" xmlns="http://www.w3.org/2000/svg"><path d="m109.52 6.827v15.285h4.007v-15.285z" style="paint-order:markers fill stroke"></path><path d="m57.641 6.382-0.87333 0.22737-0.87282 0.22738-0.8754 0.53175-0.87488 0.53175-1.5823 1.6206-0.45527 0.85421c-0.25044 0.4698-0.56288 1.2496-0.69402 1.7327l-0.23823 0.87799 0.0041 1.4102 0.0047 1.4097 0.23358 0.89038c0.12872 0.4897 0.40548 1.2437 0.61495 1.6759l0.38137 0.786 0.80615 0.87075 0.80564 0.87023 0.77153 0.49558c0.4243 0.27233 1.207 0.64375 1.7394 0.82527l0.9679 0.33176 2.9802 6e-3 0.87385-0.22737c0.4804-0.12511 1.2085-0.40868 1.6185-0.63046 0.4099-0.22181 1.1438-0.77064 1.6314-1.2196l0.88625-0.81649 0.56844-0.87591c0.31247-0.48196 0.56792-0.99973 0.56792-1.1503v-0.27389h-3.8623l-0.81546 0.66405-0.81494 0.66404-0.84181 0.24443-0.84129 0.24494-0.84026-0.0765-0.83974-0.076-0.50643-0.2558c-0.2785-0.14069-0.69618-0.43185-0.92811-0.64699-0.23193-0.21513-0.54352-0.62831-0.69246-0.91777-0.14895-0.28947-0.31692-0.77311-0.37362-1.0754l-0.10335-0.54984h11.757v-2.6086l-0.22583-0.8847c-0.12419-0.4865-0.43309-1.3102-0.68626-1.8304l-0.46044-0.94568-0.83664-0.83509-0.83716-0.83509-1.7766-0.93225-0.78652-0.16433-0.78652-0.16484zm0.46095 3.0422 1.9291 0.014 1.2154 0.63665 0.47801 0.54467c0.263 0.29955 0.5311 0.6874 0.59531 0.86197 0.0642 0.17458 0.1612 0.50108 0.21549 0.72553l0.09819 0.40773h-7.2771l0.0036-0.18552c0.0018-0.10202 0.12755-0.45955 0.27957-0.79427 0.15201-0.33473 0.50755-0.83984 0.79013-1.1224 0.28262-0.28259 0.77437-0.64291 1.093-0.80098z" style="paint-order:markers fill stroke"></path><path d="m23.966 6.3242-0.89038 0.26406c-0.4898 0.14567-1.2762 0.46943-1.7472 0.71934l-0.85628 0.45424-0.92449 0.98392-0.92449 0.9834-0.40308 0.74879c-0.22177 0.4118-0.5079 1.1447-0.63562 1.6283l-0.23203 0.87902 0.0016 1.4842 0.0016 1.4841 0.23048 0.87799c0.12671 0.4831 0.44525 1.283 0.70797 1.7772l0.47749 0.89814 1.633 1.6583 0.84026 0.47698 0.84026 0.47645 1.0888 0.28526 1.0888 0.28577 1.9415 0.016 0.8847-0.22583c0.4865-0.12417 1.3185-0.43567 1.849-0.69195l0.96428-0.4656 1.9296-1.9554 0.47129-0.9648c0.2594-0.53049 0.51706-1.1481 0.57258-1.3725l0.10129-0.40825h-3.9915l-0.154 0.33797c-0.08468 0.18586-0.34297 0.56246-0.57413 0.83716l-0.42065 0.49971-0.74776 0.3483-0.74827 0.34829h-1.9296l-1.5007-0.69866-1.0857-1.2692-0.25528-0.75861-0.25476-0.7581 0.01705-1.0423 0.01705-1.0428 0.30851-0.82424 0.30799-0.82424 0.51883-0.60565 0.51832-0.60616 0.63975-0.30954 0.64027-0.31006h2.0986l0.79633 0.41238 0.79582 0.41186 0.51263 0.77514 0.51211 0.77618h4.0711l-0.15193-0.55655c-0.08358-0.306-0.31899-0.90012-0.52348-1.3198-0.20449-0.4197-0.67978-1.1152-1.0563-1.5456l-0.68471-0.78238-1.7766-1.1482-2.0898-0.63769z" style="paint-order:markers fill stroke"></path><path d="m7.9938 6.2446-1.2144 0.14572-1.2144 0.14625-1.0831 0.52761-1.0826 0.52762-0.93896 0.8785-0.93948 0.8785-0.53382 1.016c-0.29357 0.5587-0.63541 1.4073-0.76016 1.8862l-0.22686 0.87126v1.1353c0 0.6247 0.06617 1.4957 0.14728 1.9353l0.14779 0.79943 0.59118 1.203 0.59118 1.2035 2.0547 1.9384 0.94206 0.46406 0.94258 0.46354 0.96428 0.20877c0.53048 0.11488 1.1985 0.21249 1.4841 0.21549 0.2857 4e-3 1.0481-0.0874 1.6945-0.20257l1.1751-0.20981 0.9493-0.46715 0.94878-0.46716 0.91829-0.86144 0.91777-0.86145 0.46819-0.79427c0.25755-0.43669 0.51404-0.94776 0.56947-1.1358l0.10077-0.3421-1.9792 0.03-1.9787 0.0295-0.58239 0.49712c-0.32033 0.27347-0.87206 0.63779-1.2263 0.80926l-0.6444 0.3116-1.7694 0.1602-0.67851-0.18965-0.67851-0.19017-1.1989-0.86816-0.34158-0.76895c-0.18794-0.42287-0.38376-0.95221-0.43512-1.1767l-0.09302-0.40825h11.922l-0.09767-1.4469c-0.05357-0.7958-0.18711-1.7809-0.29662-2.189-0.10951-0.4081-0.37378-1.1091-0.58756-1.558l-0.38861-0.81649-1.3405-1.5255-0.81442-0.54105c-0.448-0.29735-1.1003-0.64242-1.4495-0.76688-0.3493-0.12446-1.1568-0.28629-1.7947-0.36019zm0.24495 3.0737 0.60772 0.16381 0.60772 0.16382 0.61133 0.44287c0.33612 0.24349 0.71946 0.60753 0.85163 0.80925 0.13217 0.20173 0.33166 0.67055 0.44338 1.0418l0.20309 0.67489h-7.4259l0.10284-0.33383c0.05666-0.18364 0.20226-0.5339 0.32349-0.77876 0.12123-0.24485 0.43013-0.66811 0.68626-0.94051 0.25615-0.2724 0.72032-0.62696 1.032-0.78755l0.56689-0.29198 0.69453-0.0822z" style="paint-order:markers fill stroke"></path><path d="m101.58 6.2218-0.74207 0.093-0.74208 0.0935-0.81597 0.36535-0.81597 0.36484-1.3488 1.324 7e-3 -0.78083 7e-3 -0.78032-3.9326-0.0827v7.6471l-5.1e-4 7.6471h4.007l5.1e-4 -4.7863 5.2e-4 -4.7858 0.16381-0.59376 0.16382-0.59325 0.55759-0.53175 0.55707-0.53175 0.68213-0.1912 0.68213-0.19069 0.63613 0.0961 0.63666 0.0966 1.1865 0.66766 0.37104 0.66352 0.37155 0.66301 0.0491 5.0085 0.0491 5.009h3.9987l-0.0899-11.65-0.38912-0.82992-0.38964-0.82992-0.72347-0.69764-0.72399-0.69711-1.7808-0.87643-0.81597-0.15503z" style="paint-order:markers fill stroke"></path><path d="m123.84 6.2167-0.96532 0.0904c-0.53059 0.0499-1.3291 0.19275-1.7746 0.31781-0.4454 0.12509-1.1465 0.39776-1.558 0.60565l-0.74827 0.37776-1.9296 1.9658-0.48628 0.96428c-0.26743 0.5305-0.58352 1.3314-0.7028 1.7797l-0.21704 0.81545-2e-3 2.683 0.22531 0.88419c0.1239 0.48649 0.44105 1.3186 0.70486 1.849l0.47956 0.96428 0.89038 0.8909 0.89039 0.89039 0.97978 0.4811c0.5388 0.26433 1.3737 0.57545 1.8552 0.69195l0.87385 0.21239 1.9291 0.0206 0.92914-0.21911c0.5109-0.12047 1.3448-0.42915 1.8531-0.68574l0.92397-0.46612 1.9565-1.9637 0.46509-0.9648c0.25573-0.53051 0.50799-1.1481 0.56069-1.3725l0.0961-0.40825h-3.8628l-0.39791 0.67955-0.3979 0.67903-0.66973 0.42478-0.66921 0.42426-0.71934 0.15451-0.71933 0.154-0.84336-0.1695-0.84336-0.16898-1.3715-0.94258-0.49661-0.95136-0.4961-0.95137 0.0589-1.558 0.0584-1.5586 0.38705-0.74208c0.21297-0.40813 0.6106-0.95534 0.88315-1.2165 0.27258-0.26114 0.74351-0.57831 1.047-0.70486l0.5519-0.23048h1.6816l0.57516 0.1726 0.57465 0.17208 0.56068 0.48007c0.30836 0.26394 0.69058 0.72065 0.84905 1.0149l0.28784 0.53485h3.9765l-0.0961-0.48214c-0.053-0.2653-0.29387-0.90483-0.53537-1.4211l-0.43925-0.93845-0.93482-0.93534-0.93535-0.93483-0.95239-0.4811-0.9524-0.4806-1.1937-0.21394z" style="paint-order:markers fill stroke"></path><path d="m82.98 6.1764-1.1581 0.17984-1.1576 0.17983-1.2118 0.58653-1.2113 0.58653-0.89038 0.87643-0.88987 0.87643-0.5395 0.96428-0.53898 0.96429-0.31368 1.2413-0.31419 1.2408 0.0863 1.1891 0.0863 1.1886 0.29972 0.86403c0.16492 0.4751 0.53692 1.2532 0.82631 1.7296l0.52607 0.86609 1.7694 1.6382 0.93069 0.45372c0.512 0.24949 1.2983 0.5549 1.7472 0.67851l0.81597 0.22479 1.3358-0.0563 1.3353-0.0558 0.89039-0.33641c0.4898-0.18509 1.2244-0.55073 1.6325-0.81236s1.0928-0.82867 1.5214-1.2604c0.42856-0.4317 0.97831-1.0858 1.2216-1.4531 0.24332-0.3673 0.58586-1.0422 0.76119-1.4996l0.31885-0.832 0.0729-1.4857 0.0729-1.4857-0.17312-0.83199-0.1726-0.83251-0.51935-1.0836-0.51935-1.0842-1.9906-1.9544-2.2583-1.1152-1.1963-0.17467zm1.0919 3.7104 0.66766 0.30231 0.66817 0.30231 0.49506 0.50539c0.27247 0.27787 0.61564 0.73906 0.76223 1.0247 0.14659 0.2857 0.34707 0.91106 0.44597 1.3901l0.17983 0.87126-0.0992 0.6873c-0.0545 0.378-0.25119 1.0238-0.43718 1.4356-0.18599 0.41184-0.56447 1.0071-0.84129 1.3224l-0.50333 0.57309-0.74311 0.34624-0.7431 0.34571h-1.7808l-1.4841-0.7028-0.4718-0.50023c-0.25961-0.2752-0.65369-0.86357-0.8754-1.3074l-0.40308-0.80719-0.06718-0.97358-0.06666-0.9741 0.63872-1.6919 0.68006-0.75706 0.68058-0.75654 1.2206-0.63459z" style="paint-order:markers fill stroke"></path><path d="m68.858 0.44545v21.667h4.1553v-21.667z" style="paint-order:markers fill stroke"></path><path d="m34.707 0.44545 0.03566 10.722c0.01965 5.8971 0.06127 10.772 0.0925 10.833l0.05684 0.11007h3.8411v-9.295l0.22324-0.7524 0.22376-0.75241 1.1679-1.0552 0.53485-0.10025c0.2943-0.0553 0.8964-0.0707 1.3379-0.0341l0.80254 0.0672 0.70383 0.4346 0.70332 0.43512 0.37879 0.73639 0.3793 0.73639 0.08372 9.5064 1.85 0.0408 1.8505 0.0424-0.0026-4.9387-0.0026-4.9392-0.21497-1.0144c-0.1182-0.5578-0.37819-1.3702-0.57774-1.8061l-0.36277-0.79271-0.6072-0.65939-0.6072-0.65785-0.92759-0.42633-0.92708-0.42478-3.1161-4e-3 -1.4004 0.64699-0.74672 0.69815-0.74672 0.69866v-8.0543z" style="paint-order:markers fill stroke"></path><path d="m111.1 0-0.60306 0.2992-0.60359 0.29869-0.29662 0.40618-0.29662 0.40566v2.0779l0.65836 0.92294 0.48679 0.23978 0.48731 0.24029h1.187l0.62218-0.28215 0.62167-0.28267 0.32194-0.63045 0.32143-0.63045-0.1292-1.5596-0.41651-0.5457-0.416-0.54571-0.49506-0.2067c-0.27228-0.11382-0.71004-0.20722-0.97255-0.20722z" style="paint-order:markers fill stroke"></path></svg>`;
      this.parent.appendChild(a);
    }

    this.tooltipElement = document.createElement('div');
    this.tooltipElement.className = 'echelonic-map__tooltip';
    this.parent.appendChild(this.tooltipElement);
	
    this._setupHistoryModal();
  }

  _setupHistoryModal() {
    const modal = document.createElement('div');
    this.historyModal = modal;
    modal.className = 'echelonic-map__modal-overlay hidden';

    const content = document.createElement('div');
    content.className = 'modal__content';

    content.innerHTML = `
      <h3 class="modal__title">View Historical Heatmap</h3>
      <div class="modal__input-group">
        <label for="echelonic-start-date" class="modal__label">Start Date</label>
        <input type="date" id="echelonic-start-date" class="modal__input">
      </div>
      <div class="modal__input-group">
        <label for="echelonic-end-date" class="modal__label">End Date</label>
        <input type="date" id="echelonic-end-date" class="modal__input">
      </div>
    `;

    const btnContainer = document.createElement('div');
    btnContainer.className = 'modal__actions';

    const confirmBtn = document.createElement('button');
    confirmBtn.textContent = 'Confirm';
    confirmBtn.className = 'modal__button modal__button--confirm';
    confirmBtn.onclick = () => this._handleHistoryConfirm();

    const cancelBtn = document.createElement('button');
    cancelBtn.textContent = 'Cancel';
    cancelBtn.className = 'modal__button modal__button--cancel';
    cancelBtn.onclick = () => { modal.classList.add('hidden'); };

    btnContainer.appendChild(cancelBtn);
    btnContainer.appendChild(confirmBtn);
    content.appendChild(btnContainer);
    modal.appendChild(content);
    this.parent.appendChild(modal);

    const today = new Date();
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);
    const formatDate = (d) => d.toISOString().split('T')[0];

    content.querySelector('#echelonic-end-date').value = formatDate(today);
    content.querySelector('#echelonic-start-date').value = formatDate(today);
  }

  _glInit() {
    const gl = this.canvas.getContext('webgl', { alpha: false, antialias: true, preserveDrawingBuffer: false });
    if (!gl) throw new Error('WebGL not available');
    this.gl = gl;

    const vsWorld = `
      attribute vec3 a_pos; attribute vec4 a_col;
      uniform float u_cos, u_sin, u_ct, u_st, u_f, u_zoom;
      uniform vec2  u_pan, u_view; varying vec4 v_color;
      void main() {
        float xr = a_pos.x * u_cos - a_pos.y * u_sin, yr = a_pos.x * u_sin + a_pos.y * u_cos, z = a_pos.z;
        float y2 = u_ct * yr - u_st * z, z2 = -u_st * yr - u_ct * z;
        float denom = max(1e-3, u_f + z2);
        float sx = u_pan.x + u_zoom * (u_f * xr) / denom, sy = u_pan.y + u_zoom * (u_f * y2) / denom;
        gl_Position = vec4((2.0*sx/u_view.x)-1.0, 1.0-(2.0*sy/u_view.y), 0.0, 1.0);
        v_color = a_col;
      }`;

    const fsFlat = `precision mediump float; varying vec4 v_color; void main(){ gl_FragColor = v_color; }`;

    const vsOverlay = `
      attribute vec2 a_uv;
      uniform vec2 u_origin, u_size;
      uniform float u_cos, u_sin, u_ct, u_st, u_f, u_zoom;
      uniform vec2  u_pan, u_view; varying vec2 v_uv;
      void main() {
        float x = u_origin.x + u_size.x * a_uv.x, y = u_origin.y + u_size.y * a_uv.y;
        float xr = u_cos * x - u_sin * y, yr = u_sin * x + u_cos * y;
        float y2 = u_ct * yr, z2 = -u_st * yr;
        float denom = max(1e-3, u_f + z2);
        float sx = u_pan.x + u_zoom * (u_f * xr) / denom, sy = u_pan.y + u_zoom * (u_f * y2) / denom;
        gl_Position = vec4((2.0*sx/u_view.x)-1.0, 1.0-(2.0*sy/u_view.y), 0.0, 1.0);
        v_uv = a_uv;
      }`;

    const fsOverlay = `
      precision mediump float; varying vec2 v_uv;
      uniform sampler2D u_tex; uniform float u_opacity;
      void main(){ vec4 c = texture2D(u_tex, v_uv); gl_FragColor = vec4(c.rgb, c.a * u_opacity); }`;
    
    const vsScreenQuad = `
      attribute vec3 a_pos; attribute vec2 a_uv; attribute vec4 a_col;
      uniform float u_cos, u_sin, u_ct, u_st, u_f, u_zoom;
      uniform vec2 u_pan, u_view; uniform float u_radius_px;
      varying vec2 v_uv; varying vec4 v_color;
      void main() {
        float xr = a_pos.x * u_cos - a_pos.y * u_sin, yr = a_pos.x * u_sin + a_pos.y * u_cos;
        float z2 = -u_st * yr, y2 = u_ct * yr;
        float denom = max(1e-3, u_f + z2);
        float center_sx = u_pan.x + u_zoom * (u_f * xr) / denom;
        float center_sy = u_pan.y + u_zoom * (u_f * y2) / denom;
        float final_sx = center_sx + a_uv.x * u_radius_px;
        float final_sy = center_sy + a_uv.y * u_radius_px;
        gl_Position = vec4((2.0*final_sx/u_view.x)-1.0, 1.0-(2.0*final_sy/u_view.y), 0.0, 1.0);
        v_uv = a_uv; v_color = a_col;
      }`;
      
    const fsRadial = `
      precision mediump float;
      varying vec2 v_uv;
      varying vec4 v_color; // This will be the BORDER color from the movement_state
      uniform float u_border_thickness;
      uniform vec4 u_fill_color; // This is the inner fill color

      void main(){
        float d = length(v_uv);
        if (d > 1.0) discard;

        float AA = 0.08; // Anti-aliasing width for smooth edges
        float innerRadius = 1.0 - u_border_thickness;

        // Create an anti-aliased ring shape for the border
        float ringAlpha = smoothstep(1.0, 1.0 - AA, d) - smoothstep(innerRadius, innerRadius - AA, d);
        
        // Create an anti-aliased circle for the inner fill
        float fillAlpha = smoothstep(innerRadius, innerRadius - AA, d);
        
        // Blend the border color and the fill color together
        vec4 finalColor = mix(u_fill_color, v_color, ringAlpha);
        
        // The final pixel's opacity is the combination of the two shapes
        float finalAlpha = max(ringAlpha * v_color.a, fillAlpha * u_fill_color.a);

        gl_FragColor = vec4(finalColor.rgb, finalAlpha);
      }`;

    const fsIntensity = `
      precision mediump float; varying vec2 v_uv; varying vec4 v_color;
      void main(){
        float d = length(v_uv); if (d > 1.0) discard;
        float alpha = (1.0 - smoothstep(0.7, 1.0, d)) * v_color.a;
        gl_FragColor = vec4(alpha, 0.0, 0.0, 1.0);
      }`;

    const vsFullscreen = `attribute vec2 a_pos; varying vec2 v_uv; void main() { v_uv = a_pos * 0.5 + 0.5; gl_Position = vec4(a_pos, 0.0, 1.0); }`;
      
    const fsColorizeHeat = `
      precision mediump float; varying vec2 v_uv;
      uniform sampler2D u_intensityTex, u_gradientTex; uniform float u_opacity;
      void main() {
        float i = texture2D(u_intensityTex, v_uv).r; if (i == 0.0) discard;
        vec4 c = texture2D(u_gradientTex, vec2(i, 0.5));
        gl_FragColor = vec4(c.rgb, c.a * u_opacity);
      }`;

    const fsBlur = `
      precision mediump float; varying vec2 v_uv;
      uniform sampler2D u_image; uniform vec2 u_resolution, u_direction;
      void main() {
        vec2 texel = 1.0 / u_resolution; vec4 color = vec4(0.0);
        float weights[5];
        weights[0] = 0.227027; weights[1] = 0.1945946; weights[2] = 0.1216216;
        weights[3] = 0.054054; weights[4] = 0.016216;
        color += texture2D(u_image, v_uv) * weights[0];
        for (int i=1; i<5; ++i) {
          color += texture2D(u_image, v_uv + vec2(texel * float(i) * u_direction)) * weights[i];
          color += texture2D(u_image, v_uv - vec2(texel * float(i) * u_direction)) * weights[i];
        }
        gl_FragColor = color;
      }`;

    // Compile programs
    this._progFlat = this._makeProgram(vsWorld, fsFlat);
    this._progOverlay = this._makeProgram(vsOverlay, fsOverlay);
    this._progScreenRadial = this._makeProgram(vsScreenQuad, fsRadial);
    this._progHeatIntensity = this._makeProgram(vsScreenQuad, fsIntensity);
    this._progColorizeHeat = this._makeProgram(vsFullscreen, fsColorizeHeat);
    this._progBlur = this._makeProgram(vsFullscreen, fsBlur);

    // Locations
    this._loc = {
      flat: { a_pos: gl.getAttribLocation(this._progFlat, 'a_pos'), a_col: gl.getAttribLocation(this._progFlat, 'a_col'), u_cos: gl.getUniformLocation(this._progFlat, 'u_cos'), u_sin: gl.getUniformLocation(this._progFlat, 'u_sin'), u_ct:  gl.getUniformLocation(this._progFlat, 'u_ct'), u_st:  gl.getUniformLocation(this._progFlat, 'u_st'), u_f:   gl.getUniformLocation(this._progFlat, 'u_f'), u_zoom:gl.getUniformLocation(this._progFlat, 'u_zoom'), u_pan: gl.getUniformLocation(this._progFlat, 'u_pan'), u_view:gl.getUniformLocation(this._progFlat, 'u_view'), },
      overlay: { a_uv:  gl.getAttribLocation(this._progOverlay, 'a_uv'), u_origin: gl.getUniformLocation(this._progOverlay, 'u_origin'), u_size:   gl.getUniformLocation(this._progOverlay, 'u_size'), u_cos: gl.getUniformLocation(this._progOverlay, 'u_cos'), u_sin: gl.getUniformLocation(this._progOverlay, 'u_sin'), u_ct:  gl.getUniformLocation(this._progOverlay, 'u_ct'), u_st:  gl.getUniformLocation(this._progOverlay, 'u_st'), u_f:   gl.getUniformLocation(this._progOverlay, 'u_f'), u_zoom:gl.getUniformLocation(this._progOverlay, 'u_zoom'), u_pan: gl.getUniformLocation(this._progOverlay, 'u_pan'), u_view:gl.getUniformLocation(this._progOverlay, 'u_view'), u_tex: gl.getUniformLocation(this._progOverlay, 'u_tex'), u_opacity: gl.getUniformLocation(this._progOverlay, 'u_opacity'), },
      screenRadial: { 
        a_pos: gl.getAttribLocation(this._progScreenRadial, 'a_pos'), 
        a_uv:  gl.getAttribLocation(this._progScreenRadial, 'a_uv'), 
        a_col: gl.getAttribLocation(this._progScreenRadial, 'a_col'), 
        u_cos: gl.getUniformLocation(this._progScreenRadial, 'u_cos'), 
        u_sin: gl.getUniformLocation(this._progScreenRadial, 'u_sin'), 
        u_ct:  gl.getUniformLocation(this._progScreenRadial, 'u_ct'), 
        u_st:  gl.getUniformLocation(this._progScreenRadial, 'u_st'), 
        u_f:   gl.getUniformLocation(this._progScreenRadial, 'u_f'), 
        u_zoom:gl.getUniformLocation(this._progScreenRadial, 'u_zoom'), 
        u_pan: gl.getUniformLocation(this._progScreenRadial, 'u_pan'), 
        u_view:gl.getUniformLocation(this._progScreenRadial, 'u_view'), 
        u_radius_px: gl.getUniformLocation(this._progScreenRadial, 'u_radius_px'),
        u_border_thickness: gl.getUniformLocation(this._progScreenRadial, 'u_border_thickness'),
        u_fill_color: gl.getUniformLocation(this._progScreenRadial, 'u_fill_color')
      },
      heatIntensity: { a_pos: gl.getAttribLocation(this._progHeatIntensity, 'a_pos'), a_uv:  gl.getAttribLocation(this._progHeatIntensity, 'a_uv'), a_col: gl.getAttribLocation(this._progHeatIntensity, 'a_col'), u_cos: gl.getUniformLocation(this._progHeatIntensity, 'u_cos'), u_sin: gl.getUniformLocation(this._progHeatIntensity, 'u_sin'), u_ct:  gl.getUniformLocation(this._progHeatIntensity, 'u_ct'), u_st:  gl.getUniformLocation(this._progHeatIntensity, 'u_st'), u_f:   gl.getUniformLocation(this._progHeatIntensity, 'u_f'), u_zoom:gl.getUniformLocation(this._progHeatIntensity, 'u_zoom'), u_pan: gl.getUniformLocation(this._progHeatIntensity, 'u_pan'), u_view:gl.getUniformLocation(this._progHeatIntensity, 'u_view'), u_radius_px: gl.getUniformLocation(this._progHeatIntensity, 'u_radius_px'), },
      colorizeHeat: { a_pos: gl.getAttribLocation(this._progColorizeHeat, 'a_pos'), u_intensityTex: gl.getUniformLocation(this._progColorizeHeat, 'u_intensityTex'), u_gradientTex: gl.getUniformLocation(this._progColorizeHeat, 'u_gradientTex'), u_opacity: gl.getUniformLocation(this._progColorizeHeat, 'u_opacity'), },
      blur: { a_pos: gl.getAttribLocation(this._progBlur, 'a_pos'), u_image: gl.getUniformLocation(this._progBlur, 'u_image'), u_resolution: gl.getUniformLocation(this._progBlur, 'u_resolution'), u_direction: gl.getUniformLocation(this._progBlur, 'u_direction'), }
    };

    this._buffers.overlayQuad = gl.createBuffer();
    gl.bindBuffer(gl.ARRAY_BUFFER, this._buffers.overlayQuad);

    const subdivisions = this.options.overlaySubdivisions || 16;
    const uv_coords = [];
    for (let y = 0; y < subdivisions; y++) {
      for (let x = 0; x < subdivisions; x++) {
        const u1 = x / subdivisions;
        const v1 = y / subdivisions;
        const u2 = (x + 1) / subdivisions;
        const v2 = (y + 1) / subdivisions;

        uv_coords.push(u1, v1);
        uv_coords.push(u2, v1);
        uv_coords.push(u1, v2);

        uv_coords.push(u2, v1);
        uv_coords.push(u2, v2);
        uv_coords.push(u1, v2);
      }
    }
    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(uv_coords), gl.STATIC_DRAW);
    

    this._buffers.marker = gl.createBuffer();
    gl.bindBuffer(gl.ARRAY_BUFFER, this._buffers.marker);
    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array([-1,-1,0, 1,-1,0, -1,1,0, 1,-1,0, 1,1,0, -1,1,0]), gl.STATIC_DRAW);
    this._buffers.heatQuad = this._buffers.marker;

    this._buffers.poly = gl.createBuffer(); this._buffers.polyColor = gl.createBuffer();
    this._buffers.side = gl.createBuffer(); this._buffers.sideColor = gl.createBuffer();

    this._heatGradientTex = this._createGradientTexture(gl);
    this._fullscreenQuadBuffer = gl.createBuffer();
    gl.bindBuffer(gl.ARRAY_BUFFER, this._fullscreenQuadBuffer);
    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array([-1,-1, 1,-1, -1,1, 1,-1, 1,1, -1,1]), gl.STATIC_DRAW);
    this._heatFBO = gl.createFramebuffer();
    this._heatIntensityTex = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, this._heatIntensityTex);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE); gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR); gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
    this._blurredIntensityTex = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, this._blurredIntensityTex);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE); gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR); gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
    gl.bindTexture(gl.TEXTURE_2D, null);

    this._addEventListeners();
    this._resize();
  }

  _makeProgram(vsSrc, fsSrc) {
    const gl = this.gl;
    const vs = gl.createShader(gl.VERTEX_SHADER);
    gl.shaderSource(vs, vsSrc); gl.compileShader(vs);
    if (!gl.getShaderParameter(vs, gl.COMPILE_STATUS)) throw new Error(gl.getShaderInfoLog(vs));

    const fs = gl.createShader(gl.FRAGMENT_SHADER);
    gl.shaderSource(fs, fsSrc); gl.compileShader(fs);
    if (!gl.getShaderParameter(fs, gl.COMPILE_STATUS)) throw new Error(gl.getShaderInfoLog(fs));

    const prog = gl.createProgram();
    gl.attachShader(prog, vs); gl.attachShader(prog, fs); gl.linkProgram(prog);
    if (!gl.getProgramParameter(prog, gl.LINK_STATUS)) throw new Error(gl.getProgramInfoLog(prog));
    return prog;
  }

  setOptions(newOptions) {
    this.options = { ...this.options, ...newOptions };
    if (typeof newOptions.tilt === 'number') this.tilt = this._clampTilt(newOptions.tilt);
    if (typeof newOptions.rotation === 'number') this.rotation = newOptions.rotation;
    if (typeof newOptions.zoom === 'number') this.zoom = newOptions.zoom;

    if (newOptions.mapOverlay) {
      this.updateMapOverlay(newOptions.mapOverlay);
    }
    this._draw();
  }

  addLayer(name, url, userOptions = {}) {
    // Merge user options with defaults to ensure keys exist
    const options = { fitToGeofences: true, ...userOptions };

    if (this.layers[name]) {
      console.warn(`A layer with the name '${name}' already exists. Overwriting.`);
    }
    const newLayer = new Layer(name, url, this);
    this.layers[name] = newLayer;

    // THE FIX: This condition now ensures fitting is enabled unless explicitly set to false.
    if (options.fitToGeofences !== false) {
      newLayer._shouldFitToGeofences = true;
      newLayer._fitOptions = options.fitOptions || { padding: 1.0 };
    }

    this._loadLayerImage(newLayer);

    if (!this.activeLayer) {
      this.showLayer(name);
    }
    
    return newLayer;
  }

  showLayer(name) {
    if (this.layers[name]) {
      this.activeLayer = this.layers[name];
      this._draw();
    } else {
      console.error(`Layer '${name}' not found.`);
    }
  }

  hideActiveLayer() {
    this.activeLayer = null;
    this._draw();
  }
  
  updateMapOverlay(config = {}) {
    if (!this.activeLayer) {
        if (config.url) {
            const newLayer = this.addLayer(config.name || 'default', config.url);
            Object.assign(newLayer, config);
            this.showLayer(newLayer.name);
        } else {
            console.warn("updateMapOverlay called, but no active layer to update.");
        }
        return;
    }

    const oldUrl = this.activeLayer.url;
    Object.assign(this.activeLayer, config);

    if (config.url && config.url !== oldUrl) {
      this._loadLayerImage(this.activeLayer);
    } else {
      this._recalcOverlaySizeIfNeeded(this.activeLayer);
      this._draw();
    }
  }

  fitOverlayToGeofences({ padding = 1.0, align = 'center' } = {}, targetLayer = this.activeLayer) {
    if (!targetLayer || !targetLayer.loaded) {
      return; 
    }
    
    const geoBounds = { minX: Infinity, minY: Infinity, maxX: -Infinity, maxY: -Infinity };
    let hasGeoBounds = false;
    this.geofences.forEach(g => {
        g.coordinates?.forEach(c => {
            const p = this._geoToPixel(c[0], c[1]);
            geoBounds.minX = Math.min(geoBounds.minX, p.x);
            geoBounds.minY = Math.min(geoBounds.minY, p.y);
            geoBounds.maxX = Math.max(geoBounds.maxX, p.x);
            geoBounds.maxY = Math.max(geoBounds.maxY, p.y);
            hasGeoBounds = true;
        });
    });

    if (!hasGeoBounds) {
        console.warn("fitOverlayToGeofences called, but no geofences are present to fit to.");
        return;
    }
    
    geoBounds.width = geoBounds.maxX - geoBounds.minX;
    geoBounds.height = geoBounds.maxY - geoBounds.minY;
    geoBounds.centerX = geoBounds.minX + geoBounds.width / 2;
    geoBounds.centerY = geoBounds.minY + geoBounds.height / 2;

    if (geoBounds.width <= 0 || geoBounds.height <= 0) return;

    // --- CORRECTED SCALING LOGIC ---
    const imageAspectRatio = targetLayer.naturalW / Math.max(1, targetLayer.naturalH);
    const boundsAspectRatio = geoBounds.width / geoBounds.height;

    let newWidth, newHeight;

    if (imageAspectRatio > boundsAspectRatio) {
        // Image is wider than the bounds, so width is the limiting dimension
        newWidth = geoBounds.width * padding;
        newHeight = newWidth / imageAspectRatio;
    } else {
        // Image is taller or same proportion, so height is the limiting dimension
        newHeight = geoBounds.height * padding;
        newWidth = newHeight * imageAspectRatio;
    }

    targetLayer.width = newWidth;
    targetLayer.height = newHeight;

    if (align === 'center') {
        targetLayer.x = geoBounds.centerX - targetLayer.width / 2;
        targetLayer.y = geoBounds.centerY - targetLayer.height / 2;
    } else if (align === 'top-right'){
        targetLayer.x = geoBounds.maxX - targetLayer.width;
        targetLayer.y = geoBounds.minY;
    } else if (align === 'bottom-right') {
        targetLayer.x = geoBounds.maxX - targetLayer.width;
        targetLayer.y = geoBounds.maxY - targetLayer.height;
    } else if (align === 'bottom-left') {
        targetLayer.x = geoBounds.minX;
        targetLayer.y = geoBounds.maxY - targetLayer.height;
    } else { // 'top-left'
        targetLayer.x = geoBounds.minX;
        targetLayer.y = geoBounds.minY;
    }
    
    this._draw();
  }
  
  
  setFov(deg) {
    const min = this.options.minFovDeg, max = this.options.maxFovDeg;
    if (isFinite(deg)) {
      this.options.fovDeg = Math.max(min, Math.min(max, deg));
      this._draw();
    }
  }

  setPerspectiveEnabled(enabled) { this.options.perspective = !!enabled; this._draw(); }

  setRoofOpacity(opacity){ this.options.roofOpacity = this._clamp(opacity ?? 0, 0, 1); this._draw(); }

  setOverlaySize({width, height, lockAspect} = {}){
    const o = this.activeLayer;
    if (!o) return;
    if (typeof lockAspect === 'boolean') o.lockAspect = lockAspect;
    if (isFinite(width))  o.width  = Math.max(1e-6, width);
    if (isFinite(height)) o.height = Math.max(1e-6, height);
    this._recalcOverlaySizeIfNeeded(o);
    this._draw();
  }

  scaleOverlay(factor = 1.0, pivotWorld = null){
    const o = this.activeLayer;
    if (!o || !o.img) { this._draw(); return; }
    const ar = o.naturalW / Math.max(1, o.naturalH);
    let w = (o.width ?? (o.height ? o.height * ar : o.naturalW));
    let h = (o.height ?? (o.width  ? o.width  / ar : o.naturalH));
    if (!isFinite(w) || !isFinite(h)) return;

    factor = Math.max(1e-6, factor);
    const cx = o.x + w * 0.5, cy = o.y + h * 0.5;
    const p  = pivotWorld && isFinite(pivotWorld.x) && isFinite(pivotWorld.y) ? pivotWorld : { x: cx, y: cy };
    const nw = w * factor, nh = h * factor;
    o.x = p.x + (o.x - p.x) * factor;
    o.y = p.y + (o.y - p.y) * factor;
    o.width = nw; o.height = nh;
    this._draw();
  }

  scaleOverlayAtScreen(factor = 1.0, sx = this.canvas.width/2, sy = this.canvas.height/2){
    const pivot = this.screenToWorld(sx, sy);
    this.scaleOverlay(factor, pivot);
  }

  async init() {
    try {
      this._showMessage('Loading map...');
      await this._fetchGeofences();
      this._resolveGeofencesReady(); // NEW: Signal that geofences are ready for use.

      if (this.options.mapOverlay && this.options.mapOverlay.url) {
        const legacyLayer = this.addLayer('default', this.options.mapOverlay.url);
        Object.assign(legacyLayer, this.options.mapOverlay);
      }
      
      this.fitGeofencesInView();
      
      await this._fetchTagUpdates();
      this._scheduleNextUpdate();
      this._draw();
      this._showMessage('Map initialized!');
      const a = this._getRotationAnchorScreen();
      this.rotateBy(-0.78, a.x, a.y);
	  this.canvas.classList.remove("hidden");
    } catch (e) {
      console.error(e);
      this._showMessage('Init failed: ' + e.message, 4000, 'error');
    }
  }

  _addEventListeners() {
    const c = this.canvas;
	c.addEventListener('contextmenu', e => e.preventDefault()); 
    c.addEventListener('mousedown', this._onMouseDown);
    c.addEventListener('mousemove', this._onMouseMove);
    c.addEventListener('mouseup', this._onMouseUpOrLeave);
    c.addEventListener('mouseleave', this._onMouseUpOrLeave);
    c.addEventListener('wheel', this._onWheel, { passive: false });
    c.addEventListener('touchstart', this._onTouchStart, { passive: false });
    c.addEventListener('touchmove', this._onTouchMove, { passive: false });
    c.addEventListener('touchend', this._onTouchEnd, { passive: false });
    window.addEventListener('keydown', this._onKeyDown);
    window.addEventListener('resize', () => this._resize());
  }

   _onMouseDown = (e) => {
    this.isAnimatingPan = false;
    this.canvas.classList.add('grabbing');

    if (e.button === 2 || (e.button === 0 && (e.ctrlKey || e.altKey))) {
      this.isRotatingAndTilting = true;
      this.isPanning = false;
      this.dragStart = { 
          x: e.clientX, y: e.clientY, 
          rotation: this.rotation, tilt: this.tilt, 
          zoom: this.zoom // <-- ADD THIS LINE
      };
      const anchorScreen = this._getRotationAnchorScreen();
      this.dragStart.worldAnchor = this.screenToWorld(anchorScreen.x, anchorScreen.y);
      this.dragStart.screenAnchor = anchorScreen;
    } 
    // --- CONDITION FOR PAN ---
    // This is the default action for the Left Mouse Button.
    else if (e.button === 0) {
      this.isPanning = true;
      this.isRotatingAndTilting = false;
      this.lastPos = { x: e.clientX, y: e.clientY };
    }
  }

  _onMouseMove = (e) => {
    this._handleTooltip(e);

    if (this.isRotatingAndTilting && this.dragStart) {
        const rotatePerPxRad = -0.35 * Math.PI / 180;
        const tiltPerPxRad = 0.30 * Math.PI / 180;
        const dx = e.clientX - this.dragStart.x;
        const dy = e.clientY - this.dragStart.y;

        // Get the screen position of our anchor point BEFORE applying the new transforms
        const posBefore = this.worldToScreen(this.dragStart.worldAnchor.x, this.dragStart.worldAnchor.y);

        // Apply the new rotation and tilt based on the mouse drag
        if (this.options.enableRotate) {
            this.rotation = this._normalizeAngle(this.dragStart.rotation + dx * rotatePerPxRad);
        }
        if (this.options.enableTilt) {
            this.tilt = this._clampTilt(this.dragStart.tilt - dy * tiltPerPxRad);
        }

        // --- START: UNIFIED STABILIZATION LOGIC ---
        // Get the screen position of our anchor point AFTER the new transforms
        const posAfter = this.worldToScreen(this.dragStart.worldAnchor.x, this.dragStart.worldAnchor.y);

        // Calculate how the anchor's distance from the view center changed
        const distBefore = Math.hypot(posBefore.x - this.panOffset.x, posBefore.y - this.panOffset.y);
        const distAfter = Math.hypot(posAfter.x - this.panOffset.x, posAfter.y - this.panOffset.y);
        
        let zoomCorrectionFactor = 1.0;
        if (distAfter > 1e-3) {
            // Create a counter-zoom to cancel out the distance change
            zoomCorrectionFactor = distBefore / distAfter;
        }

        // Apply the correction based on the zoom level at the start of the drag
        // this.zoom = this.dragStart.zoom * zoomCorrectionFactor;
        // --- END: UNIFIED STABILIZATION LOGIC ---

        // With all transforms and zoom corrected, do the final pan to lock the anchor point
        this._panToKeepWorldAtScreen(
            this.dragStart.worldAnchor.x,
            this.dragStart.worldAnchor.y,
            this.dragStart.screenAnchor.x,
            this.dragStart.screenAnchor.y
        );
        
        this._draw();
    } 
    else if (this.isPanning) {
        const dx = e.clientX - this.lastPos.x;
        const dy = e.clientY - this.lastPos.y;
        this.panOffset.x += dx;
        this.panOffset.y += dy;
        this.lastPos = { x: e.clientX, y: e.clientY };
        this._draw();
    }
  }

   _onMouseUpOrLeave = () => {
    this.isPanning = false; this.isRotatingAndTilting = false; this.dragStart = null;
    this.lastPos = null; // <-- ADD THIS LINE
    this.canvas.classList.remove('grabbing');
    if (this.tooltipElement) {
        this.tooltipElement.style.display = 'none';
    }
  }

 _onWheel = (e) => {
    e.preventDefault();
    const rect = this.canvas.getBoundingClientRect();
    const cx = e.clientX - rect.left, cy = e.clientY - rect.top;
 if (this.options.enableZoom) {
            const wasZoomIn = e.deltaY < 0; 
            this.zoomAt(cx, cy, wasZoomIn ? 1.03 : 0.97);
        }
}

  _onTouchStart = (e) => {
    this.isAnimatingPan = false;
    e.preventDefault();

    if (e.touches.length === 1) {
        this.isPanning = true;
        this.isRotatingAndTilting = false;
        const touch = e.touches[0];
        this.lastPos = { x: touch.clientX, y: touch.clientY };
        this.dragStart = null; 
    } else if (e.touches.length === 2) {
        this.isPanning = false;
        this.isRotatingAndTilting = false;
        this.dragStart = null;
        this.initialPinch = this._pinchDistance(e.touches);
        this.initialRotateAngle = this._pinchAngle(e.touches);

        const rect = this.canvas.getBoundingClientRect();
        const midX = (e.touches[0].clientX + e.touches[1].clientX) / 2;
        const midY = (e.touches[0].clientY + e.touches[1].clientY) / 2;
        const anchorScreen = { x: midX - rect.left, y: midY - rect.top };

        const anchorWorld = this.screenToWorld(anchorScreen.x, anchorScreen.y);
        this._gestureAnchorWorld = anchorWorld;
        this._gestureAnchorScreen = anchorScreen;

        this.initialTilt = this.tilt;
        this.initialMidY = midY - rect.top;
    } else if (e.touches.length === 3) {
        this.isPanning = true;
        this.isRotatingAndTilting = false;
        this.dragStart = null;
        this.lastPos = { x: e.touches[0].clientX, y: e.touches[0].clientY };
    }
}

  _onTouchMove = (e) => {
    e.preventDefault();
    if (this.isPanning && (e.touches.length === 1 || e.touches.length === 3)) {
        const currentPos = { x: e.touches[0].clientX, y: e.touches[0].clientY };
        const dx = currentPos.x - this.lastPos.x;
        const dy = currentPos.y - this.lastPos.y;
        this.panOffset.x += dx;
        this.panOffset.y += dy;
        this.lastPos = currentPos;
        this._draw();
    } 
    else if (e.touches.length === 1 && this.isRotatingAndTilting) {
        const rotatePerPxRad = -0.35 * Math.PI / 180;
        const tiltPerPxRad = 0.30 * Math.PI / 180;
        const touch = e.touches[0];
        const dx = touch.clientX - this.dragStart.x, dy = touch.clientY - this.dragStart.y;
        if (this.options.enableRotate) this.setRotation(this.dragStart.rotation + dx * rotatePerPxRad);
        if (this.options.enableTilt) this.setTilt(this.dragStart.tilt - dy * tiltPerPxRad);
    } else if (e.touches.length === 2) {
        const rect = this.canvas.getBoundingClientRect();
        const cy = (e.touches[0].clientY + e.touches[1].clientY)/2 - rect.top;
        if (this.options.enableZoom) {
            const dist = this._pinchDistance(e.touches);
            if (this.initialPinch) {
                this.zoom = Math.max(0.1, Math.min(100, this.zoom * (dist/this.initialPinch)));
                this.initialPinch = dist;
            }
        }
        if (this.options.enableRotate) {
            const ang = this._pinchAngle(e.touches);
            if (this.initialRotateAngle != null) {
                let delta = ang - this.initialRotateAngle;
                delta = ((delta + Math.PI) % (2*Math.PI)) - Math.PI;
                this.rotation = this._normalizeAngle(this.rotation + delta);
                this.initialRotateAngle = ang;
            }
        }
        if (this.options.enableTilt) {
            if (this.initialMidY != null && this.initialTilt != null) {
                const dy = cy - this.initialMidY;
                this.setTilt(this.initialTilt - dy * (this.options.tiltTouchSensitivity||0.003));
            }
        }
        const w = this._gestureAnchorWorld || this._getBoundsCenterWorld() || this.screenToWorld(this.canvas.width/2, this.canvas.height/2);
        const s = this._gestureAnchorScreen || this.worldToScreen(w.x, w.y);
        this._panToKeepWorldAtScreen(w.x, w.y, s.x, s.y);
        this._draw();
    }
}

  _onTouchEnd = () => {
    this.isPanning = false;
    this.isRotatingAndTilting = false;
    this.dragStart = null;
    this.lastPos = null;
    this.initialPinch = null;
    this.initialRotateAngle = null;
    this.initialTilt = null;
    this.initialMidY = null;
    this._gestureAnchorScreen = null;
    this._gestureAnchorWorld = null;
    this.canvas.classList.remove('grabbing');
  }

  _onKeyDown = (e) => {
    if (['INPUT','TEXTAREA','SELECT','BUTTON'].includes((e.target.tagName || ''))) return;
    const rotStep = Math.PI / 36;
    const anchor = this._getRotationAnchorScreen();
    switch (e.key.toLowerCase()) {
      case 'q': if (this.options.enableRotate) this.rotateBy(-rotStep, anchor.x, anchor.y); break;
      case 'e': if (this.options.enableRotate) this.rotateBy(+rotStep, anchor.x, anchor.y); break;
      case 'w': if (this.options.enableTilt) this.setTilt(this.tilt + this.options.tiltStep); break;
      case 's': if (this.options.enableTilt) this.setTilt(this.tilt - this.options.tiltStep); break;
    }
  }

   _resize() {
    const rect = this.parent.getBoundingClientRect();
    const w = Math.max(1, rect.width|0), h = Math.max(1, rect.height|0);
    if (this.canvas.width !== w || this.canvas.height !== h) {
      this.canvas.width = w; this.canvas.height = h;
      const gl = this.gl;
      if (gl) {
        if (this._heatIntensityTex) {
          gl.bindTexture(gl.TEXTURE_2D, this._heatIntensityTex);
          gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, w, h, 0, gl.RGBA, gl.UNSIGNED_BYTE, null);
        }
        if (this._blurredIntensityTex) {
          gl.bindTexture(gl.TEXTURE_2D, this._blurredIntensityTex);
          gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, w, h, 0, gl.RGBA, gl.UNSIGNED_BYTE, null);
        }
        gl.bindTexture(gl.TEXTURE_2D, null);
      }
    }
    this._draw();
  }

  _pinchDistance(t) { const dx=t[0].clientX-t[1].clientX, dy=t[0].clientY-t[1].clientY; return Math.hypot(dx,dy); }
  _pinchAngle(t) { const dx=t[1].clientX-t[0].clientX, dy=t[1].clientY-t[0].clientY; return Math.atan2(dy,dx); }
  _normalizeAngle(a) { return ((a % (2*Math.PI)) + (2*Math.PI)) % (2*Math.PI); }
  _clampTilt(t) { return Math.max(0, Math.min(this.options.maxTilt, t || 0)); }
  _clamp(v,a,b){ return Math.max(a, Math.min(b, v)); }
  _lerp(a,b,t){ return a + (b-a) * t; }

  _autoAdjustFov() {
    if (!this.options.perspective || !this.options.autoFov || this.tilt <= 1e-6 || !this.geofences.length) return;
    const s = Math.sin(this.tilt); if (Math.abs(s) < 1e-5) return;
    const c = Math.cos(this.rotation), si = Math.sin(this.rotation);
    let minYr = Infinity, maxYr = -Infinity;
    for (const g of this.geofences) {
      for (const v of (g.coordinates || [])) {
        const w = this._geoToPixel(v[0], v[1]);
        const yr = si * w.x + c * w.y;
        minYr = Math.min(minYr, yr); maxYr = Math.max(maxYr, yr);
      }
    }
    if (!isFinite(minYr) || !isFinite(maxYr) || maxYr - minYr < 1e-6) return;
    const target = Math.max(1.001, this.options.autoFovTargetRatio || 1.3);
    const a = s * minYr, b = s * maxYr;
    let f = (target * b - a) / (target - 1);
    const minF = (b + 8.0);
    if (!isFinite(f) || f < minF) f = minF;
    const h = Math.max(1, this.canvas.height);
    const fovRad = 2 * Math.atan((0.5 * h) / Math.max(1e-3, f));
    let fovDeg = this._clamp(fovRad * 180 / Math.PI, this.options.minFovDeg || 25, this.options.maxFovDeg || 100);
    const cur = this.options.fovDeg || 45;
    const t = this._clamp(this.options.autoFovSmoothing ?? 0.25, 0, 1);
    this.options.fovDeg = this._lerp(cur, fovDeg, t);
  }

  _getFocalPx() {
    const fov = (this.options.fovDeg || 45) * Math.PI / 180;
    const h = Math.max(1, this.canvas.height || 1);
    return (0.5 * h) / Math.tan(Math.max(0.01, fov * 0.5));
  }

  worldToScreen(x, y) {
    const c = Math.cos(this.rotation), s = Math.sin(this.rotation);
    const xr = c*x - s*y, yr = s*x + c*y;
    if (this.options.perspective && this.tilt > 1e-6) {
      const ct = Math.cos(this.tilt), st = Math.sin(this.tilt);
      const y2 = ct * yr, z2 = -st * yr;
      const f = this._getFocalPx();
      const denom = Math.max(1e-3, f + z2);
      const projx = (f * xr) / denom, projy = (f * y2) / denom;
      return { x: this.panOffset.x + this.zoom * projx, y: this.panOffset.y + this.zoom * projy };
    } else {
      const ky = Math.cos(this.tilt) || 1e-6;
      return { x: this.panOffset.x + this.zoom * xr, y: this.panOffset.y + this.zoom * ky * yr };
    }
  }

  screenToWorld(x, y) {
    const c = Math.cos(this.rotation), s = Math.sin(this.rotation);
    const sdx = (x - this.panOffset.x) / this.zoom, sdy = (y - this.panOffset.y) / this.zoom;
    if (this.options.perspective && this.tilt > 1e-6) {
      const ct = Math.cos(this.tilt), st = Math.sin(this.tilt);
      const f = this._getFocalPx();
      const denomY = (f * ct + sdy * st);
      const yr = (Math.abs(denomY) < 1e-6) ? 0 : (sdy * f) / denomY;
      const xr = ((sdx * (f - st * yr)) / f);
      return { x: c * xr + s * yr, y: -s * xr + c * yr };
    } else {
      const ky = Math.cos(this.tilt) || 1e-6;
      const xr = sdx, yr = sdy / ky;
      return { x: c * xr + s * yr, y: -s * xr + c * yr };
    }
  }

  _panToKeepWorldAtScreen(wx, wy, sx, sy) {
    const proj = this.worldToScreen(wx, wy);
    this.panOffset.x += (sx - proj.x); this.panOffset.y += (sy - proj.y);
  }

  zoomAt(x,y,factor){
    const old = this.zoom, z = Math.max(0.1, Math.min(100, old * factor));
    if (z === old) return;
    const w = this.screenToWorld(x, y);
    this.zoom = z;
    this._panToKeepWorldAtScreen(w.x, w.y, x, y);
    this._draw();
  }

  rotateBy(delta, ax, ay) {
    if (ax === undefined || ay === undefined) { const a = this._getRotationAnchorScreen(); ax=a.x; ay=a.y; }
    const w = this.screenToWorld(ax, ay);
    this.rotation = this._normalizeAngle(this.rotation + delta);
    this._panToKeepWorldAtScreen(w.x, w.y, ax, ay);
    this._draw();
  }

  setRotation(angle = 0, ax, ay) {
    if (ax === undefined || ay === undefined) { const a = this._getRotationAnchorScreen(); ax=a.x; ay=a.y; }
    const w = this.screenToWorld(ax, ay);
    this.rotation = this._normalizeAngle(angle);
    this._panToKeepWorldAtScreen(w.x, w.y, ax, ay);
    this._draw();
  }

  setTilt(angle) {
    const newTilt = this._clampTilt((angle === null) ? 0 : Math.max(angle, 0.0001));
    if (Math.abs(newTilt - this.tilt) < 1e-6) return;

    // 1. Get the screen anchor and its corresponding point in the world BEFORE tilting
    const anchorScreen = this._getRotationAnchorScreen();
    const worldAnchor = this.screenToWorld(anchorScreen.x, anchorScreen.y);

    // 2. Temporarily apply the new tilt to see its effect
    const oldTilt = this.tilt;
    this.tilt = newTilt;

    // --- START: STABILIZATION LOGIC ---
    // 3. Calculate where the world anchor moves to on screen AFTER the tilt
    const projectedPosAfterTilt = this.worldToScreen(worldAnchor.x, worldAnchor.y);

    // 4. Calculate the distance change from the center of the screen
    // This tells us how much the view "shrunk" or "grew"
    const distBefore = Math.hypot(anchorScreen.x - this.panOffset.x, anchorScreen.y - this.panOffset.y);
    const distAfter = Math.hypot(projectedPosAfterTilt.x - this.panOffset.x, projectedPosAfterTilt.y - this.panOffset.y);

    // 5. Apply a zoom correction to counteract the size change
    if (distAfter > 1e-3) {
      const zoomCorrectionFactor = distBefore / distAfter;
      this.zoom *= zoomCorrectionFactor;
    }
    // --- END: STABILIZATION LOGIC ---

    // 6. Now, with the tilt and zoom corrected, do the final pan to lock the anchor point
    this._panToKeepWorldAtScreen(worldAnchor.x, worldAnchor.y, anchorScreen.x, anchorScreen.y);
    
    this._draw();
  }

  _computeCombinedBounds() {
    let finalBounds = { minX: Infinity, minY: Infinity, maxX: -Infinity, maxY: -Infinity };
    let hasBounds = false;

    if (this.geofences.length > 0) {
        this.geofences.forEach(g => {
        g.coordinates?.forEach(c => {
            const p = this._geoToPixel(c[0], c[1]);
            finalBounds.minX = Math.min(finalBounds.minX, p.x);
            finalBounds.minY = Math.min(finalBounds.minY, p.y);
            finalBounds.maxX = Math.max(finalBounds.maxX, p.x);
            finalBounds.maxY = Math.max(finalBounds.maxY, p.y);
        });
        });
        if (isFinite(finalBounds.minX)) {
            hasBounds = true;
        }
    }

    const layer = this.activeLayer;
    if (layer && layer.loaded && layer.img) {
        const w = (layer.width ?? (layer.height ? layer.height * (layer.naturalW / layer.naturalH) : layer.naturalW));
        const h = (layer.height ?? (layer.width ? layer.width / (layer.naturalW / layer.naturalH) : layer.naturalH));

        if (hasBounds) {
            finalBounds.minX = Math.min(finalBounds.minX, layer.x);
            finalBounds.minY = Math.min(finalBounds.minY, layer.y);
            finalBounds.maxX = Math.max(finalBounds.maxX, layer.x + w);
            finalBounds.maxY = Math.max(finalBounds.maxY, layer.y + h);
        } else {
            finalBounds = { minX: layer.x, minY: layer.y, maxX: layer.x + w, maxY: layer.y + h };
        }
        hasBounds = true;
    }

    if (!hasBounds) return null;

    finalBounds.width = finalBounds.maxX - finalBounds.minX;
    finalBounds.height = finalBounds.maxY - finalBounds.minY;
    finalBounds.centerX = (finalBounds.minX + finalBounds.maxX) * 0.5;
    finalBounds.centerY = (finalBounds.minY + finalBounds.maxY) * 0.5;

    return finalBounds;
  }

  _getBoundsCenterWorld(){ const b = this._computeCombinedBounds(); return b ? { x: b.centerX, y: b.centerY } : null; }
  
  _getRotationAnchorScreen() {
    return { x: this.canvas.width / 2, y: this.canvas.height / 2 };
  }
  
  _drawLines(bufVerts, bufCols, verts, cols) {
    const gl = this.gl;
    if (!verts.length) return;

    // Use the same shader as geofences
    gl.useProgram(this._progFlat);
    gl.uniform1f(this._loc.flat.u_cos, Math.cos(this.rotation));
    gl.uniform1f(this._loc.flat.u_sin, Math.sin(this.rotation));
    gl.uniform1f(this._loc.flat.u_ct,  Math.cos(this.tilt));
    gl.uniform1f(this._loc.flat.u_st,  Math.sin(this.tilt));
    gl.uniform1f(this._loc.flat.u_f,   this._getFocalPx());
    gl.uniform1f(this._loc.flat.u_zoom,this.zoom);
    gl.uniform2f(this._loc.flat.u_pan, this.panOffset.x, this.panOffset.y);
    gl.uniform2f(this._loc.flat.u_view, this.canvas.width, this.canvas.height);

    gl.bindBuffer(gl.ARRAY_BUFFER, bufVerts);
    gl.bufferData(gl.ARRAY_BUFFER, verts, gl.DYNAMIC_DRAW);
    gl.enableVertexAttribArray(this._loc.flat.a_pos);
    gl.vertexAttribPointer(this._loc.flat.a_pos, 3, gl.FLOAT, false, 0, 0);

    gl.bindBuffer(gl.ARRAY_BUFFER, bufCols);
    gl.bufferData(gl.ARRAY_BUFFER, cols, gl.DYNAMIC_DRAW);
    gl.enableVertexAttribArray(this._loc.flat.a_col);
    gl.vertexAttribPointer(this._loc.flat.a_col, 4, gl.FLOAT, false, 0, 0);

    gl.enable(gl.BLEND);
    gl.blendFunc(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);

    // This is the key difference: drawing line strips instead of triangles
    gl.drawArrays(gl.LINE_STRIP, 0, verts.length / 3);

    gl.disable(gl.BLEND);
}

_drawTagTrails() {
    if (!this.options.trailsEnabled) return;

    const baseColor = this._hexToRgb(this.options.trailColor || this.invertedTheme);
    const r = baseColor[0] / 255;
    const g = baseColor[1] / 255;
    const b = baseColor[2] / 255;

    // Lazily create buffers if they don't exist
    if (!this._buffers.trail) this._buffers.trail = this.gl.createBuffer();
    if (!this._buffers.trailColor) this._buffers.trailColor = this.gl.createBuffer();

    for (const trail of Object.values(this.tagTrails)) {
        if (trail.length < 2) continue;

        const trailVerts = [];
        const trailCols = [];

        for (let i = 0; i < trail.length; i++) {
            const p = trail[i];
            if (isNaN(p.lon) || isNaN(p.lat)) continue;

            const worldPos = this._geoToPixel(p.lon, p.lat);
            trailVerts.push(worldPos.x, worldPos.y, 0); // z=0 to draw on the floor

            // Calculate alpha to create a fading effect
            const alpha = (i / (trail.length - 1)) * 0.8; // Fade from 0 to 0.8
            trailCols.push(r, g, b, alpha);
        }

        this._drawLines(
            this._buffers.trail,
            this._buffers.trailColor,
            new Float32Array(trailVerts),
            new Float32Array(trailCols)
        );
    }
}

  fitGeofencesInView() {
    const b = this._computeCombinedBounds();
    if (!b) {
      this.zoom = 1;
      this.panOffset = { x: this.canvas.width/2, y: this.canvas.height/2 };
      this.rotation = 0;
      this._draw(); return;
    }
    const padding = 1.1;
    this.zoom = Math.min(100, Math.min(this.canvas.width/b.width, this.canvas.height/b.height)/padding);
    this.rotation = 0;
    this._panToKeepWorldAtScreen(b.centerX, b.centerY, this.canvas.width/2, this.canvas.height/2);
    this._draw();
  }

  async _fetchGeofences() {
    this._assertApi();
    const r = await fetch(`${this.options.baseUrl}/geofences`, { headers: { Authorization: 'Bearer ' + this.options.authToken } });
    if (!r.ok) throw new Error('Failed to fetch geofences');
    const data = await r.json();
    this.geofences = (data.geofences || []).map(g => {
      try { return { ...g, coordinates: JSON.parse(g.vertices) }; } catch { return { ...g, coordinates: [] }; }
    });
  }

   async _fetchTagUpdates() {
    this._assertApi();
    const r = await fetch(`${this.options.baseUrl}/tags`, { headers: { Authorization: 'Bearer ' + this.options.authToken } });
    if (!r.ok) throw new Error('Failed to fetch tags');
    const data = await r.json();
    const updates = (data.tags || []);
    this.tags = updates.filter(t => t.movement_state !== 'offline').map(t => ({
      id: t.tag_id, lon: parseFloat(t.pos_x), lat: parseFloat(t.pos_y),
      geofence_name: t.geofence_name, movement_state: t.movement_state
    }));
    if (this.options.stats) this.renderData(this.analyzeTagData(this.tags));

    const offlineTagIds = updates.filter(t => t.movement_state === 'offline').map(t => t.tag_id);

    // --- START: NEW TRAIL LOGIC ---
    if (this.options.trailsEnabled) {
      // Remove trails for offline tags
      offlineTagIds.forEach(id => {
        if (id) {
            delete this.tagTrails[id];
        }
      });
      
      // Update trails for active tags
      this.tags.forEach(tag => {
        if (!tag.id || isNaN(tag.lon) || isNaN(tag.lat)) return;
        
        const trail = this.tagTrails[tag.id] || [];
        trail.push({ lon: tag.lon, lat: tag.lat });

        // Trim the trail to the maximum length
        while (trail.length > this.options.trailLength) {
          trail.shift();
        }
        
        this.tagTrails[tag.id] = trail;
      });
    } else {
        // If trails are disabled, clear any existing data
        this.tagTrails = {};
    }
    // --- END: NEW TRAIL LOGIC ---

    if (this.options.heatmap) {
      const offlineTagIds = updates.filter(t => t.movement_state === 'offline').map(t => t.tag_id);
      
      offlineTagIds.forEach(id => { 
        if (id) {
            delete this.tagHeatmapData[id]; 
        }
      });

      this.tags.forEach(tag => {
        const history = this.tagHeatmapData[tag.id] || [];
        history.push({ lon: tag.lon, lat: tag.lat });
        this.tagHeatmapData[tag.id] = history.slice(-1);
      });
    }
  }

  _scheduleNextUpdate() {
    if (this.updateTimeout) clearTimeout(this.updateTimeout);
    if (this.isHistoryView) return;
    this.updateTimeout = setTimeout(async () => {
      try {
        await this._fetchTagUpdates();
        this._draw();
      } catch (e) { console.error('Tag update error', e); }
      this._scheduleNextUpdate();
    }, this.options.updateInterval);
  }

  async _handleHistoryConfirm() {
    const startInput = this.historyModal.querySelector('#echelonic-start-date');
    const endInput = this.historyModal.querySelector('#echelonic-end-date');
    if (!startInput.value || !endInput.value) {
      this._showMessage('Please select both a start and end date.', 2500, 'error');
      return;
    }
    if (new Date(startInput.value) > new Date(endInput.value)) {
      this._showMessage('Start date cannot be after end date.', 2500, 'error');
      return;
    }
    this.historyModal.classList.add('hidden');
    await this._fetchAndDisplayHistory(startInput.value, endInput.value);
  }

  async _fetchAndDisplayHistory(startDate, endDate) {
    this._assertApi();
    this._showMessage('Loading historical data...');
    clearTimeout(this.updateTimeout);
    this.isHistoryView = true;
    this.statsBox.classList.add('hidden');
    this.tags = [];
    this.tagHeatmapData = {};
    this._draw();
    if (this._historyBtn) this._historyBtn.classList.add('hidden');
    if (this._closeHistoryBtn) this._closeHistoryBtn.classList.remove('hidden');

    try {
      const startISO = new Date(startDate + 'T00:00:00.000Z').toISOString();
      const endISO = new Date(endDate + 'T23:59:59.999Z').toISOString();
      const url = `${this.options.baseUrl}/analytics/heatmap?startDate=${startISO}&endDate=${endISO}`;
      const r = await fetch(url, { headers: { Authorization: 'Bearer ' + this.options.authToken } });
      if (!r.ok) throw new Error(`Failed to fetch history (${r.status})`);
      const data = await r.json();
      const historyPoints = data.points || [];
      if (historyPoints.length === 0) {
        this._showMessage('No historical data found for this period.', 2500, 'ok');
        return;
      }
      this.tagHeatmapData['history'] = historyPoints.map(p => ({ lon: p.x, lat: p.y }));
      this._draw();
      this._showMessage(`Displaying ${historyPoints.length} historical points.`);
    } catch (e) {
      console.error('Failed to load history:', e);
      this._showMessage(e.message, 3000, 'error');
      this._exitHistoryView();
    }
  }
  
  _exitHistoryView() {
    if (!this.isHistoryView) return;
    this.isHistoryView = false;
    this.statsBox.classList.remove('hidden');
    this.tags = [];
    this.tagHeatmapData = {};
    this._draw();
    if (this._historyBtn) this._historyBtn.classList.remove('hidden');
    if (this._closeHistoryBtn) this._closeHistoryBtn.classList.add('hidden');
    this._showMessage('Switched to real-time view.');
    this._scheduleNextUpdate();
  }

  renderData(jsonData) {
    this.statsBoxOnline.innerHTML = `<small>${this.options.totalText}</small><b>${jsonData.online}</b>`;
    this.statsBoxGF.innerHTML = '';
    for (const roomName in jsonData.geofences) {
      if (!Object.prototype.hasOwnProperty.call(jsonData.geofences, roomName)) continue;
      const el = document.createElement('div');
      el.className = 'geofence-item';
      el.innerHTML = `<p>${roomName}</p><b class="geofence-item__count">${jsonData.geofences[roomName]}</b>`;
      this.statsBoxGF.appendChild(el);
    }
  }

  analyzeTagData(tagData) {
    const summary = { online: 0, geofences: {} };
    if (!Array.isArray(tagData)) return summary;
    tagData.forEach(tag => {
      summary.online++;
      if (tag.geofence_name) summary.geofences[tag.geofence_name] = (summary.geofences[tag.geofence_name] || 0) + 1;
    });
    return summary;
  }

  _draw() {
    const gl = this.gl; if (!gl) return;
    this._autoAdjustFov();
    gl.viewport(0, 0, this.canvas.width, this.canvas.height);
    gl.clearColor(1.0, 1.0, 1.0, 1.0); gl.clear(gl.COLOR_BUFFER_BIT);

    if (this.activeLayer && this.activeLayer.loaded) {
        this._drawOverlay();
    }

    if (this.options.geofences) {
      this._rebuildGeofenceGeometry();
      this._drawFlat(this._buffers.side, this._buffers.sideColor, this._geom.sideVerts, this._geom.sideCols, true);
      this._drawFlat(this._buffers.poly, this._buffers.polyColor, this._geom.polyVerts, this._geom.polyCols, true);
    }
    if (this.options.heatmap) this._drawHeatmap();

    // Add the call to draw trails BEFORE markers, so markers draw on top.
    if (this.options.trailsEnabled) this._drawTagTrails(); // <-- ADD THIS LINE

    if (this.options.tagMarkers) this._drawTagMarkers();
}

  _drawFlat(bufVerts, bufCols, verts, cols, blend=false) {
    const gl = this.gl; if (!verts.length) return;
    gl.useProgram(this._progFlat);
    gl.uniform1f(this._loc.flat.u_cos, Math.cos(this.rotation));
    gl.uniform1f(this._loc.flat.u_sin, Math.sin(this.rotation));
    gl.uniform1f(this._loc.flat.u_ct,  Math.cos(this.tilt));
    gl.uniform1f(this._loc.flat.u_st,  Math.sin(this.tilt));
    gl.uniform1f(this._loc.flat.u_f,   this._getFocalPx());
    gl.uniform1f(this._loc.flat.u_zoom,this.zoom);
    gl.uniform2f(this._loc.flat.u_pan, this.panOffset.x, this.panOffset.y);
    gl.uniform2f(this._loc.flat.u_view, this.canvas.width, this.canvas.height);

    gl.bindBuffer(gl.ARRAY_BUFFER, bufVerts);
    if (bufVerts._size !== verts.length) { gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(verts), gl.DYNAMIC_DRAW); bufVerts._size = verts.length; }
    else { gl.bufferSubData(gl.ARRAY_BUFFER, 0, new Float32Array(verts)); }
    gl.enableVertexAttribArray(this._loc.flat.a_pos);
    gl.vertexAttribPointer(this._loc.flat.a_pos, 3, gl.FLOAT, false, 0, 0);

    gl.bindBuffer(gl.ARRAY_BUFFER, bufCols);
    if (bufCols._size !== cols.length) { gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(cols), gl.DYNAMIC_DRAW); bufCols._size = cols.length; }
    else { gl.bufferSubData(gl.ARRAY_BUFFER, 0, new Float32Array(cols)); }
    gl.enableVertexAttribArray(this._loc.flat.a_col);
    gl.vertexAttribPointer(this._loc.flat.a_col, 4, gl.FLOAT, false, 0, 0);

    if (blend) { gl.enable(gl.BLEND); gl.blendFunc(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA); }
    else { gl.disable(gl.BLEND); }
    gl.drawArrays(gl.TRIANGLES, 0, verts.length / 3);
    if (blend) gl.disable(gl.BLEND);
  }

  _drawOverlay() {
    const gl = this.gl;
    const layer = this.activeLayer;
    if (!layer || !layer.img || !layer.texture) return;

    gl.useProgram(this._progOverlay);
    gl.uniform1f(this._loc.overlay.u_cos, Math.cos(this.rotation));
    gl.uniform1f(this._loc.overlay.u_sin, Math.sin(this.rotation));
    gl.uniform1f(this._loc.overlay.u_ct,  Math.cos(this.tilt));
    gl.uniform1f(this._loc.overlay.u_st,  Math.sin(this.tilt));
    gl.uniform1f(this._loc.overlay.u_f,   this._getFocalPx());
    gl.uniform1f(this._loc.overlay.u_zoom,this.zoom);
    gl.uniform2f(this._loc.overlay.u_pan, this.panOffset.x, this.panOffset.y);
    gl.uniform2f(this._loc.overlay.u_view, this.canvas.width, this.canvas.height);
    
    const w = (layer.width ?? (layer.height ? layer.height * (layer.naturalW/layer.naturalH) : layer.naturalW));
    const h = (layer.height ?? (layer.width ? layer.width / (layer.naturalW/layer.naturalH) : layer.naturalH));
    gl.uniform2f(this._loc.overlay.u_origin, layer.x, layer.y);
    gl.uniform2f(this._loc.overlay.u_size, w, h);
    gl.uniform1f(this._loc.overlay.u_opacity, layer.opacity);
    
    gl.activeTexture(gl.TEXTURE0);
    gl.bindTexture(gl.TEXTURE_2D, layer.texture);
    gl.uniform1i(this._loc.overlay.u_tex, 0);
    
    gl.bindBuffer(gl.ARRAY_BUFFER, this._buffers.overlayQuad);
    gl.enableVertexAttribArray(this._loc.overlay.a_uv);
    gl.vertexAttribPointer(this._loc.overlay.a_uv, 2, gl.FLOAT, false, 0, 0);
    gl.drawArrays(gl.TRIANGLES, 0, layer.vertexCount);
    gl.bindTexture(gl.TEXTURE_2D, null);
  }

  _drawHeatmap() {
    const gl = this.gl; const heatPoints = [];
    for (const h of Object.values(this.tagHeatmapData)) for (const p of h) heatPoints.push(this._geoToPixel(p.lon, p.lat));
    if (!heatPoints.length) return;

    gl.bindFramebuffer(gl.FRAMEBUFFER, this._heatFBO);
    gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, this._heatIntensityTex, 0);
    gl.viewport(0, 0, this.canvas.width, this.canvas.height);
    gl.clearColor(0,0,0,0); gl.clear(gl.COLOR_BUFFER_BIT);
    gl.enable(gl.BLEND); gl.blendFunc(gl.ONE, gl.ONE);

    gl.useProgram(this._progHeatIntensity);
    const loc1 = this._loc.heatIntensity;
    gl.uniform1f(loc1.u_cos, Math.cos(this.rotation)); gl.uniform1f(loc1.u_sin, Math.sin(this.rotation));
    gl.uniform1f(loc1.u_ct, Math.cos(this.tilt)); gl.uniform1f(loc1.u_st, Math.sin(this.tilt));
    gl.uniform1f(loc1.u_f, this._getFocalPx()); gl.uniform1f(loc1.u_zoom, this.zoom);
    gl.uniform2f(loc1.u_pan, this.panOffset.x, this.panOffset.y);
    gl.uniform2f(loc1.u_view, this.canvas.width, this.canvas.height);
    gl.uniform1f(loc1.u_radius_px, this.options.heatRadiusPx);

    const n = heatPoints.length; const verts = new Float32Array(n*6*3), uvs = new Float32Array(n*6*2), cols = new Float32Array(n*6*4);
    let vi=0, ui=0, ci=0; const uv_q = [-1,-1, 1,-1, -1,1, 1,-1, 1,1, -1,1]; const alpha = this.options.heatIntensity;
    for(const p of heatPoints){ for(let i=0;i<6;i++){ verts.set([p.x,p.y,0],vi);vi+=3; cols.set([1,1,1,alpha],ci);ci+=4; } uvs.set(uv_q,ui);ui+=12; }

    const bufP=this._buffers._heatP||(this._buffers._heatP=gl.createBuffer()); gl.bindBuffer(gl.ARRAY_BUFFER,bufP); gl.bufferData(gl.ARRAY_BUFFER,verts,gl.DYNAMIC_DRAW); gl.enableVertexAttribArray(loc1.a_pos); gl.vertexAttribPointer(loc1.a_pos,3,gl.FLOAT,false,0,0);
    const bufU=this._buffers._heatU||(this._buffers._heatU=gl.createBuffer()); gl.bindBuffer(gl.ARRAY_BUFFER,bufU); gl.bufferData(gl.ARRAY_BUFFER,uvs,gl.DYNAMIC_DRAW); gl.enableVertexAttribArray(loc1.a_uv); gl.vertexAttribPointer(loc1.a_uv,2,gl.FLOAT,false,0,0);
    const bufC=this._buffers._heatC||(this._buffers._heatC=gl.createBuffer()); gl.bindBuffer(gl.ARRAY_BUFFER,bufC); gl.bufferData(gl.ARRAY_BUFFER,cols,gl.DYNAMIC_DRAW); gl.enableVertexAttribArray(loc1.a_col); gl.vertexAttribPointer(loc1.a_col,4,gl.FLOAT,false,0,0);
    gl.drawArrays(gl.TRIANGLES, 0, n * 6); gl.disable(gl.BLEND);

    gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, this._blurredIntensityTex, 0);
    gl.clear(gl.COLOR_BUFFER_BIT);
    gl.useProgram(this._progBlur); const locB = this._loc.blur;
    gl.activeTexture(gl.TEXTURE0); gl.bindTexture(gl.TEXTURE_2D, this._heatIntensityTex);
    gl.uniform1i(locB.u_image, 0); gl.uniform2f(locB.u_resolution, this.canvas.width, this.canvas.height);
    gl.uniform2f(locB.u_direction, this.options.heatBlur, 0.0);
    gl.bindBuffer(gl.ARRAY_BUFFER, this._fullscreenQuadBuffer);
    gl.enableVertexAttribArray(locB.a_pos); gl.vertexAttribPointer(locB.a_pos, 2, gl.FLOAT, false, 0, 0);
    gl.drawArrays(gl.TRIANGLES, 0, 6);

    gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D, this._heatIntensityTex, 0);
    gl.clear(gl.COLOR_BUFFER_BIT);
    gl.bindTexture(gl.TEXTURE_2D, this._blurredIntensityTex);
    gl.uniform2f(locB.u_direction, 0.0, this.options.heatBlur);
    gl.drawArrays(gl.TRIANGLES, 0, 6);

    gl.bindFramebuffer(gl.FRAMEBUFFER, null);
    gl.enable(gl.BLEND); gl.blendFunc(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);
    gl.useProgram(this._progColorizeHeat); const locC = this._loc.colorizeHeat;
    gl.activeTexture(gl.TEXTURE0); gl.bindTexture(gl.TEXTURE_2D, this._heatIntensityTex); gl.uniform1i(locC.u_intensityTex, 0);
    gl.activeTexture(gl.TEXTURE1); gl.bindTexture(gl.TEXTURE_2D, this._heatGradientTex); gl.uniform1i(locC.u_gradientTex, 1);
    gl.uniform1f(locC.u_opacity, this.options.heatOpacity);
    gl.drawArrays(gl.TRIANGLES, 0, 6);
    gl.disable(gl.BLEND); gl.activeTexture(gl.TEXTURE0); gl.bindTexture(gl.TEXTURE_2D, null);
  }

  _drawTagMarkers() {
    if (!this.tags.length) return;
    const gl = this.gl;
    
    gl.useProgram(this._progScreenRadial); 
    const loc = this._loc.screenRadial;

    gl.uniform1f(loc.u_cos, Math.cos(this.rotation)); 
    gl.uniform1f(loc.u_sin, Math.sin(this.rotation));
    gl.uniform1f(loc.u_ct,  Math.cos(this.tilt)); 
    gl.uniform1f(loc.u_st,  Math.sin(this.tilt));
    gl.uniform1f(loc.u_f,   this._getFocalPx()); 
    gl.uniform1f(loc.u_zoom,this.zoom);
    gl.uniform2f(loc.u_pan, this.panOffset.x, this.panOffset.y);
    gl.uniform2f(loc.u_view, this.canvas.width, this.canvas.height);
    
    gl.uniform1f(loc.u_radius_px, this.options.markerRadius);
    gl.uniform1f(loc.u_border_thickness, this.options.markerBorderThickness);

    const parseColor = (colorStr) => {
        if (colorStr.startsWith('rgba')) {
            const parts = colorStr.match(/[\d.]+/g);
            return [parts[0]/255, parts[1]/255, parts[2]/255, parseFloat(parts[3])];
        }
        const rgb = this._hexToRgb(colorStr);
        return [rgb[0]/255, rgb[1]/255, rgb[2]/255, 1.0];
    };
    gl.uniform4fv(loc.u_fill_color, parseColor(this.options.markerFillColor));

    const stateColors = {
      is_moving: '#0071e3',
      started_moving: '#ff5722',
      stopped_moving: '#ff5722',
      is_stopped: '#059900',
      default: '#8e8e93'
    };

    const n = this.tags.length;
    const verts = new Float32Array(n * 6 * 3);
    const uvs   = new Float32Array(n * 6 * 2);
    const cols  = new Float32Array(n * 6 * 4);
    let vi=0, ui=0, ci=0;
    
    const uv_quad = [ -1,-1,  1,-1,  -1,1,   1,-1,  1,1,  -1,1 ];

    for (const t of this.tags) {
      if (isNaN(t.lon) || isNaN(t.lat)) continue;

      const state = t.movement_state || 'default';
      const hexColor = stateColors[state.toLowerCase()] || stateColors.default;
      
      const rgbColor = this._hexToRgb(hexColor);
      const normalizedBorderColor = [rgbColor[0] / 255, rgbColor[1] / 255, rgbColor[2] / 255, 1.0];

      const wp = this._geoToPixel(t.lon, t.lat);
      for (let k=0; k<6; k++){ 
        verts.set([wp.x, wp.y, 0], vi); vi+=3;
        cols.set(normalizedBorderColor, ci); ci+=4; 
      }
      uvs.set(uv_quad, ui); ui += 12;
    }

    const bufP = this._buffers._mkP || (this._buffers._mkP = gl.createBuffer()); 
    gl.bindBuffer(gl.ARRAY_BUFFER, bufP); gl.bufferData(gl.ARRAY_BUFFER, verts, gl.DYNAMIC_DRAW); 
    gl.enableVertexAttribArray(loc.a_pos); gl.vertexAttribPointer(loc.a_pos, 3, gl.FLOAT, false, 0, 0);
    
    const bufU = this._buffers._mkU || (this._buffers._mkU = gl.createBuffer()); 
    gl.bindBuffer(gl.ARRAY_BUFFER, bufU); gl.bufferData(gl.ARRAY_BUFFER, uvs, gl.DYNAMIC_DRAW); 
    gl.enableVertexAttribArray(loc.a_uv); gl.vertexAttribPointer(loc.a_uv, 2, gl.FLOAT, false, 0, 0);
    
    const bufC = this._buffers._mkC || (this._buffers._mkC = gl.createBuffer()); 
    gl.bindBuffer(gl.ARRAY_BUFFER, bufC); gl.bufferData(gl.ARRAY_BUFFER, cols, gl.DYNAMIC_DRAW); 
    gl.enableVertexAttribArray(loc.a_col); gl.vertexAttribPointer(loc.a_col, 4, gl.FLOAT, false, 0, 0);
    
    gl.enable(gl.BLEND);
    gl.blendFunc(gl.SRC_ALPHA, gl.ONE_MINUS_SRC_ALPHA);
    gl.drawArrays(gl.TRIANGLES, 0, n * 6);
    gl.disable(gl.BLEND);
  }

  _rebuildGeofenceGeometry() {
    const themeRGB = this._hexToRgb(this.options.theme);
    const topColor = [themeRGB[0]/255, themeRGB[1]/255, themeRGB[2]/255, this._clamp(this.options.roofOpacity??0.0, 0, 1)];
    const sideColor= [Math.max(0,themeRGB[0]-40)/255, Math.max(0,themeRGB[1]-40)/255, Math.max(0,themeRGB[2]-40)/255, 0.3];
    const pV=[], pC=[], sV=[], sC=[];
    for (const g of this.geofences) {
      if (!g.coordinates?.length) continue;
      const wPts = g.coordinates.map(c => this._geoToPixel(c[0], c[1]));
      const h = this._getExtrusionWorld(g, wPts);
      const triIdx = this._triangulateEarcut(wPts);
      for (let i=0;i<triIdx.length;i+=3) { const a=wPts[triIdx[i]], b=wPts[triIdx[i+1]], c=wPts[triIdx[i+2]]; pV.push(a.x,a.y,h, b.x,b.y,h, c.x,c.y,h); pC.push(...topColor,...topColor,...topColor); }
      const n = wPts.length;
      for (let i=0;i<n;i++) { const p1=wPts[i], p2=wPts[(i+1)%n]; sV.push(p1.x,p1.y,0, p2.x,p2.y,0, p2.x,p2.y,h, p1.x,p1.y,0, p2.x,p2.y,h, p1.x,p1.y,h); for(let k=0;k<6;k++) sC.push(...sideColor); }
    }
    this._geom.polyVerts=new Float32Array(pV); this._geom.polyCols=new Float32Array(pC);
    this._geom.sideVerts=new Float32Array(sV); this._geom.sideCols=new Float32Array(sC);
  }

  _triangulateEarcut(points) {
    const idx = [...points.keys()];
    if (this._polygonArea(points) < 0) idx.reverse();
    const result = []; let guard = 0;
    while (idx.length > 3 && guard++ < 10000) {
      let earFound = false;
      for (let i=0;i<idx.length;i++) {
        const i0 = idx[(i-1+idx.length)%idx.length], i1 = idx[i], i2 = idx[(i+1)%idx.length];
        const a=points[i0], b=points[i1], c=points[i2];
        if (this._area2(a,b,c) <= 0) continue;
        let contains = false;
        for (let j=0;j<idx.length;j++) { const k=idx[j]; if (k===i0||k===i1||k===i2) continue; if(this._pointInTri(points[k],a,b,c)) { contains=true; break; } }
        if (!contains) { result.push(i0,i1,i2); idx.splice(i,1); earFound=true; break; }
      }
      if (!earFound) break;
    }
    if (idx.length === 3) result.push(idx[0], idx[1], idx[2]);
    return result;
  }

  _polygonArea(p){ let a=0; for(let i=0,n=p.length;i<n;i++){ const c=p[i], d=p[(i+1)%n]; a+=c.x*d.y - d.x*c.y; } return a*0.5; }
  _area2(a,b,c){ return (b.x-a.x)*(c.y-a.y) - (b.y-a.y)*(c.x-a.x); }
  _pointInTri(p,a,b,c){ const x0=c.x-a.x, y0=c.y-a.y, x1=b.x-a.x, y1=b.y-a.y, x2=p.x-a.x, y2=p.y-a.y; const d00=x0*x0+y0*y0, d01=x0*x1+y0*y1, d02=x0*x2+y0*y2, d11=x1*x1+y1*y1, d12=x1*x2+y1*y2; const invD=1/Math.max(1e-9,(d00*d11-d01*d01)); const u=(d11*d02-d01*d12)*invD, v=(d00*d12-d01*d02)*invD; return (u>=0)&&(v>=0)&&(u+v<=1); }

  _getExtrusionWorld(g) {
    let baseH = this.options.extrudeHeight || 0;
    if (this.options.geofenceHeights && g.name && (g.name in this.options.geofenceHeights)) {
      baseH = this.options.geofenceHeights[g.name];
    }
    return baseH;
  }

  async _loadLayerImage(layer) {
    const img = new Image();
    img.crossOrigin = 'anonymous';
    try {
        await new Promise((res, rej) => { img.onload = res; img.onerror = rej; img.src = layer.url; });
        
        layer.img = img;
        layer.naturalW = img.naturalWidth;
        layer.naturalH = img.naturalHeight;
        
        const gl = this.gl;
        layer.texture = gl.createTexture();
        gl.bindTexture(gl.TEXTURE_2D, layer.texture);
        gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, 0);
        gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, img);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE); 
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
        gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
        gl.bindTexture(gl.TEXTURE_2D, null);

        layer.loaded = true;

        if (layer._shouldFitToGeofences) {
          await this._geofencesReadyPromise;
          this.fitOverlayToGeofences(layer._fitOptions, layer);
        } else {
          this._recalcOverlaySizeIfNeeded(layer);
        }

        // --- THE FIX ---
        // If this is the very first layer added to the map, its bounds are now known.
        // We should re-fit the global view to ensure both geofences and the layer are visible.
        if (Object.keys(this.layers).length === 1) {
          this.fitGeofencesInView();
        } else {
          this._draw(); // For subsequent layers, a simple redraw is enough.
        }
        
    } catch (e) {
        console.error(`Failed to load image for layer '${layer.name}' from URL: ${layer.url}`, e);
    }
  }

  _recalcOverlaySizeIfNeeded(layer) {
    if (!layer || !layer.img) return;
    if (layer.lockAspect && layer.width && !layer.height) {
        layer.height = layer.width * (layer.naturalH / layer.naturalW);
    } else if (layer.lockAspect && layer.height && !layer.width) {
        layer.width = layer.height * (layer.naturalW / layer.naturalH);
    }
  }

  _geoToPixel(lon, lat) { return { x: lon, y: lat }; }
  _hexToRgb(h) { const m=/^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(h); return m?[parseInt(m[1],16),parseInt(m[2],16),parseInt(m[3],16)]:[0,0,0]; }
  invertHexColor(hex, invertToBlackOrWhite = false) {
    if (!hex || typeof hex !== 'string') return '#000000';
    let c = hex.startsWith('#')?hex.slice(1):hex; if(c.length===3)c=c.split('').map(i=>i+i).join('');
    if (c.length !== 6) return '#000000';
    const r=parseInt(c.slice(0,2),16), g=parseInt(c.slice(2,4),16), b=parseInt(c.slice(4,6),16);
    if(invertToBlackOrWhite) return (0.299*r+0.587*g+0.114*b)>128?'#000000':'#FFFFFF';
    const inv = v=>(255-v).toString(16).padStart(2,'0'); return `#${inv(r)}${inv(g)}${inv(b)}`;
  }

  _injectControls() {
    if (this._controlsEl && this._controlsEl.parentNode) this._controlsEl.parentNode.removeChild(this._controlsEl);
    const container = this.parent;
    const controls = document.createElement('div');
    controls.className = 'echelonic-map__controls';

    const makeBtn = (svg, onClick, title) => {
      const btn = document.createElement('button');
      btn.className = 'controls__button';
      btn.innerHTML = svg;
      if (title) btn.title = title;
      btn.addEventListener('click', onClick);
      return btn;
    };

    const makeGroup = () => {
        const g = document.createElement('div');
        g.className = 'controls__group';
        return g;
    };

    const cx = () => (this.canvas ? this.canvas.width/2 : 0);
    const cy = () => (this.canvas ? this.canvas.height/2 : 0);
    const rotAnchor = () => (typeof this._getRotationAnchorScreen==='function')?this._getRotationAnchorScreen():{ x: cx(), y: cy() };

    const buttons = {
      zoomIn: makeBtn('<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="#000000"><path d="M440-440H200v-80h240v-240h80v240h240v80H520v240h-80v-240Z"/></svg>', () => this.zoomAt(cx(), cy(), 1.2), 'Zoom in'),
	  zoomOut: makeBtn('<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px" fill="#000000"><path d="M200-440v-80h560v80H200Z"/></svg>', () => (this.zoomAt(cx(), cy(), 0.8)), 'Zoom out'),
      fit: makeBtn('<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px"><path d="M480-320q-66 0-113-47t-47-113q0-66 47-113t113-47q66 0 113 47t47 113q0 66-47 113t-113 47Zm0-80q33 0 56.5-23.5T560-480q0-33-23.5-56.5T480-560q-33 0-56.5 23.5T400-480q0 33 23.5 56.5T480-400Zm0-80ZM200-120q-33 0-56.5-23.5T120-200v-160h80v160h160v80H200Zm400 0v-80h160v-160h80v160q0 33-23.5 56.5T760-120H600ZM120-600v-160q0-33 23.5-56.5T200-840h160v80H200v160h-80Zm640 0v-160H600v-80h160q33 0 56.5 23.5T840-760v160h-80Z"/></svg>', () => { const a=rotAnchor(); this.setRotation(0,a.x,a.y); this.setTilt(null); this.fitGeofencesInView(); }, 'Reset view'),
      history: makeBtn('<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px"><path d="M480-120q-138 0-240.5-91.5T122-440h82q14 104 92.5 172T480-200q117 0 198.5-81.5T760-480q0-117-81.5-198.5T480-760q-69 0-129 32t-101 88h110v80H120v-240h80v94q51-64 124.5-99T480-840q75 0 140.5 28.5t114 77q48.5 48.5 77 114T840-480q0 75-28.5 140.5t-77 114q-48.5 48.5-114 77T480-120Zm112-192L440-464v-216h80v184l128 128-56 56Z"/></svg>', () => { this.historyModal.classList.remove('hidden'); }, 'View history'),
      closeHistory: makeBtn('<svg xmlns="http://www.w3.org/2000/svg" height="24px" viewBox="0 -960 960 960" width="24px"><path d="m785-289-58-58q16-29 24.5-63t8.5-70q0-117-81.5-198.5T480-760q-35 0-68.5 8.5T348-726l-59-59q43-26 91.5-40.5T480-840q75 0 140.5 28.5t114 77q48.5 48.5 77 114T840-480q0 53-14.5 101T785-289ZM520-554l-80-80v-46h80v126ZM792-56 672-176q-42 26-90 41t-102 15q-138 0-240.5-91.5T122-440h82q14 104 92.5 172T480-200q37 0 70.5-8.5T614-234L288-560H120v-168l-64-64 56-56 736 736-56 56Z"/></svg>', () => this._exitHistoryView(), 'Close history view')
    };

    this._historyBtn = buttons.history;
    this._closeHistoryBtn = buttons.closeHistory;
    this._closeHistoryBtn.classList.add('controls__button--inverted', 'hidden');

    if (this.options.enableZoom) { const g=makeGroup(); g.append(buttons.zoomIn, buttons.zoomOut); controls.append(g); }
    if (this.options.enableTopDown) { const g=makeGroup(); g.append(buttons.fit); controls.append(g); }
    if (this.options.baseUrl && this.options.authToken) { const g=makeGroup(); g.classList.add('controls__group--push-to-bottom'); g.append(buttons.history, buttons.closeHistory); controls.append(g); }

    container.appendChild(controls);
    this._controlsEl = controls;
    return controls;
  }

  _assertApi() { if (!this.options.baseUrl || !this.options.authToken) throw new Error('baseUrl and authToken are required'); }
  _showMessage(msg, ms = 1600, type = 'ok') {
    this.messageBox.textContent = msg;
    this.messageBox.classList.toggle('error', type === 'error');
    this.messageBox.classList.add('visible');
    clearTimeout(this._msgTimer);
    this._msgTimer = setTimeout(() => { this.messageBox.classList.remove('visible'); }, ms);
  }

}